/*
 * @(#)DrawingEntry.java    3.0 2001/01/31
 *
 * Luca Cabibbo 
 * Dipartimento di Informatica e Automazione 
 * Universit degli Studi Roma Tre 
 * cabibbo@dia.uniroma3.it 
 */ 

package fiji.grafica; 

import java.awt.*; 
import java.awt.image.ImageObserver; 

/** 
 * Un oggetto <code>DrawingEntry</code>  un
 * oggetto grafico disegnato in una DrawingFrame. 
 * 
 * @version 3.0 
 * @author Luca Cabibbo 
 */ 
class DrawingEntry { 
    
    static final int CLEARRECT = 1; 
    static final int CLIPRECT  = 2; 
    static final int COPYAREA  = 3; 
    static final int DRAW3DRECT = 4; 
    static final int DRAWARC = 5; 
    static final int DRAWBYTES = 6; 
    static final int DRAWCHARS = 7; 
    static final int DRAWIMAGE1 = 8; 
    static final int DRAWIMAGE2 = 9; 
    static final int DRAWIMAGE3 = 10; 
    static final int DRAWIMAGE4 = 11; 
    static final int DRAWIMAGE5 = 12; 
    static final int DRAWIMAGE6 = 13; 
    static final int DRAWLINE = 14; 
    static final int DRAWOVAL = 15; 
    static final int DRAWPOLYGON1 = 16; 
    static final int DRAWPOLYGON2 = 17; 
    static final int DRAWPOLYLINE = 18; 
    static final int DRAWRECT = 19; 
    static final int DRAWROUNDRECT = 20; 
    static final int DRAWSTRING1 = 21; 
    static final int DRAWSTRING2 = 22; 
    static final int FILL3DRECT = 23; 
    static final int FILLARC = 24; 
    static final int FILLOVAL = 25; 
    static final int FILLPOLYGON1 = 26; 
    static final int FILLPOLYGON2 = 27; 
    static final int FILLRECT = 28; 
    static final int FILLROUNDRECT = 29; 
    static final int SETCOLOR = 30; 
    static final int SETFONT = 31; 
    static final int TRANSLATE = 32; 
    
    /** Tipo della entry grafica. */ 
    private int type; 
    
    /* Parametri della entry grafica. */ 
    private int x, y; 
    private int width, height; 
    private int dx, dy; 
    private boolean raised; 
    private String str; 
    private byte[] bytedata; 
    private char[] chardata; 
    private int offset, length; 
    private Color color; 
    private Font font; 
    private Image img; 
    private ImageObserver observer; 
    private int[] xPoints; 
    private int[] yPoints; 
    private int nPoints; 
    private Polygon p; 
    
    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type) { 
        this.type = type; 
    }
    
    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, int x, int y) { 
        this(type); 
        this.x = x; 
        this.y = y; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, int x, int y, int width, int height) { 
        this(type, x, y); 
        this.width = width; 
        this.height = height; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, int x, int y, int width, int height, boolean raised) { 
        this(type, x, y, width, height); 
        this.raised = raised; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, int x, int y, int width, int height, int dx, int dy) { 
        this(type, x, y, width, height); 
        this.dx = dx; 
        this.dy = dy; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, String str, int x, int y) { 
        this(type, x, y); 
        this.str = str; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, byte[] data, int offset, int length, int x, int y) { 
        this(type, x, y); 
        this.bytedata = data; 
        this.offset = offset; 
        this.length = length; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, char[] data, int offset, int length, int x, int y) { 
        this(type, x, y); 
        this.chardata = data; 
        this.offset = offset; 
        this.length = length; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, Image img, int x, int y, Color bgcolor, ImageObserver observer) { 
        this(type, x, y); 
        this.img = img; 
        this.color = bgcolor; 
        this.observer = observer; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, Image img, int x, int y, ImageObserver observer) { 
        this(type, x, y); 
        this.img = img; 
        this.observer = observer; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, Image img, int x, int y, int width, int height, Color bgcolor, ImageObserver observer) { 
        this(type, x, y); 
        this.img = img; 
        this.width = width; 
        this.height = height; 
        this.color = bgcolor; 
        this.observer = observer; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, Image img, int x, int y, int width, int height, ImageObserver observer) { 
        this(type, x, y); 
        this.img = img; 
        this.width = width; 
        this.height = height; 
        this.observer = observer; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, int[] xPoints, int[] yPoints, int nPoints) { 
        this(type); 
        this.xPoints = xPoints; 
        this.yPoints = yPoints; 
        this.nPoints = nPoints; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, Polygon p) { 
        this(type); 
        this.p = p; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, Color c) { 
        this(type); 
        this.color = c; 
    }

    /** Crea una nuova DrawingEntry. */ 
    DrawingEntry(int type, Font f) { 
        this(type); 
        this.font = f; 
    }

    /** 
     * Disegna la entry grafica nel contesto grafico g. 
     */ 
    void draw(Graphics g) { 
        int x1, y1, x2, y2; 
        int startAngle, arcAngle; 
        int arcWidth, arcHeight; 
        Color bgcolor; 
        switch (type) {
            case CLEARRECT: 
                g.clearRect(x, y, width, height); 
                break; 
            case CLIPRECT: 
                g.clipRect(x, y, width, height); 
                break; 
            case COPYAREA: 
                g.copyArea(x, y, width, height, dx, dy); 
                break; 
            case DRAW3DRECT: 
                g.draw3DRect(x, y, width, height, raised); 
                break; 
            case DRAWARC: 
                startAngle = dx; 
                arcAngle = dy; 
                g.drawArc(x, y, width, height, startAngle, arcAngle); 
                break; 
            case DRAWBYTES: 
                g.drawBytes(bytedata, offset, length, x, y); 
                break; 
            case DRAWCHARS: 
                g.drawChars(chardata, offset, length, x, y); 
                break; 
            case DRAWIMAGE1: 
                bgcolor = color; 
                g.drawImage(img, x, y, bgcolor, observer); 
                break; 
            case DRAWIMAGE2: 
                g.drawImage(img, x, y, observer); 
                break; 
            case DRAWIMAGE3: 
                bgcolor = color; 
                g.drawImage(img, x, y, width, height, bgcolor, observer); 
                break; 
            case DRAWIMAGE4: 
                g.drawImage(img, x, y, width, height, observer); 
                break; 
            case DRAWIMAGE5: 
                break; 
            case DRAWIMAGE6: 
                break; 
            case DRAWLINE: 
                x1 = x; y1 = y; 
                x2 = width; y2 = height; 
                g.drawLine(x1, y1, x2, y2); 
                break; 
            case DRAWOVAL: 
                g.drawOval(x, y, width, height); 
                break; 
            case DRAWPOLYGON1: 
                g.drawPolygon(xPoints, yPoints, nPoints); 
                break; 
            case DRAWPOLYGON2: 
                g.drawPolygon(p); 
                break; 
            case DRAWPOLYLINE: 
                g.drawPolyline(xPoints, yPoints, nPoints); 
                break; 
            case DRAWRECT: 
                g.drawRect(x, y, width, height); 
                break; 
            case DRAWROUNDRECT: 
                arcWidth = dx; 
                arcHeight = dy; 
                g.drawRoundRect(x, y, width, height, arcWidth, arcHeight); 
                break; 
            case DRAWSTRING1: 
                break; 
            case DRAWSTRING2: 
                g.drawString(str, x, y); 
                break; 
            case FILL3DRECT: 
                g.fill3DRect(x, y, width, height, raised); 
                break; 
            case FILLARC: 
                startAngle = dx; 
                arcAngle = dy; 
                g.fillArc(x, y, width, height, startAngle, arcAngle); 
                break; 
            case FILLOVAL: 
                g.fillOval(x, y, width, height); 
                break; 
            case FILLPOLYGON1: 
                g.fillPolygon(xPoints, yPoints, nPoints); 
                break; 
            case FILLPOLYGON2: 
                g.fillPolygon(p); 
                break; 
            case FILLRECT: 
                g.fillRect(x, y, width, height); 
                break; 
            case FILLROUNDRECT: 
                arcWidth = dx; 
                arcHeight = dy; 
                g.fillRoundRect(x, y, width, height, arcWidth, arcHeight); 
                break; 
            case SETCOLOR: 
                g.setColor(color); 
                break; 
            case SETFONT: 
                g.setFont(font); 
                break; 
            case TRANSLATE: 
                g.translate(x, y); 
                break; 
        } 
    }
} 
