/*
 * @(#)DrawingFrame.java    3.0 2001/01/31
 *
 * Luca Cabibbo 
 * Dipartimento di Informatica e Automazione 
 * Universit degli Studi Roma Tre 
 * cabibbo@dia.uniroma3.it 
 */ 

package fiji.grafica; 

import java.awt.Color; 
import java.awt.Font; 
import java.awt.Graphics; 
import java.awt.Image; 
import java.awt.Polygon; 
import java.awt.image.ImageObserver; 

import javax.swing.*; 

/** 
 * Un oggetto <code>DrawingFrame</code>  una finestra 
 * grafica su cui  possibile disegnare e scrivere. 
 * 
 * Un oggetto <code>DrawingFrame</code> incapsula 
 * le propriet&agrave; necessarie per il disegno di elementi 
 * grafici, tra cui un sistema di coordinate e 
 * un colore corrente. 
 * 
 * Un oggetto <code>DrawingFrame</code> 
 * sa eseguire operazioni che <i>disegnano</i> figure 
 * (<code>draw...</code>, che disegnano il contorno delle figure) 
 * e operazioni che <i>riempiono</i> figure 
 * (<code>fill...</code>, che disegnano le figure a colore pieno). 
 * 
 * Il sistema di coordinate &egrave; organizzato come segue: 
 * la prima coordinata &egrave; una coordinata orizzontale, 
 * che varia da sinistra verso destra; 
 * la seconda coordinata &egrave; una coordinata verticale, 
 * che varia dall'alto verso il basso. 
 * Le coordinate sono intere. 
 * Inizialmente la coordinata <code>(0,0)</code> indica 
 * l'angolo in alto a sinistra dell'area di disegno. 
 * Il sistema di coordinate pu&ograve; 
 * essere modificato utilizzando l'operazione <code>translate</code>. 
 * 
 * @version 3.0 
 * @author Luca Cabibbo 
 */ 
public class DrawingFrame extends JFrame { 

    /** Pannello per il disegno degli elementi grafico. */ 
    private DrawingPanel panel; 
    
    /** 
     * Crea una <code>DrawingFrame</code>, 
     * di titolo <code>titolo</code>, 
     * di larghezza <code>width</code> 
     * e altezza <code>height</code> punti. 
     */ 
    public DrawingFrame(String titolo, int width, int height) { 
        super(titolo); 
        panel = new DrawingPanel(width, height); 
        JPanel p = new JPanel();
        this.getContentPane().add(p);
        p.setSize(width, height);
        p.setLayout(null);
        p.add(panel);
        panel.setBackground(Color.white);
        p.setBackground(Color.blue);
        //this.setSize(width+10,height+30);
        this.setSize(this.getPreferredSize());
        this.setVisible(true); 

        Graphics g = panel.getGraphics(); 
        this.setColor(g.getColor()); 
        this.setFont(g.getFont());

        /* gestione eventi */ 
        this.addWindowListener( new DrawingFrameAdapter() ); 
    } 
    
    /** Ridisegna la finestra grafica. */ 
    public void repaint(){
        panel.repaint();
    }
    
    void clearRect(int x, int y, int width, int height) { 
        addEntry( new DrawingEntry(DrawingEntry.CLEARRECT, x, y, width, height) ); 
    } 

    void clipRect(int x, int y, int width, int height) { 
        addEntry( new DrawingEntry(DrawingEntry.CLIPRECT, x, y, width, height) ); 
    } 

    void copyArea(int x, int y, int width, int height, int dx, int dy) { 
        addEntry( new DrawingEntry(DrawingEntry.CLIPRECT, x, y, width, height, dx, dy) ); 
    } 

    void draw3DRect(int x, int y, int width, int height, boolean raised) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAW3DRECT, x, y, width, height, raised) ); 
    } 

    /** 
     * Disegna un arco circolare o ellittico che copre 
     * il rettangolo caratterizzato dai parametri 
     * <code>x</code>, <code>y</code>, 
     * <code>width</code> e <code>height</code>.
     */ 
    public void drawArc(int x, int y, int width, int height, int startAngle, int arcAngle) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWARC, x, y, width, height, startAngle, arcAngle) ); 
    } 

    void drawBytes(byte[] data, int offset, int length, int x, int y) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWBYTES, data, offset, length, x, y) ); 
    } 

    void drawChars(char[] data, int offset, int length, int x, int y) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWCHARS, data, offset, length, x, y) ); 
    } 

    void drawImage(Image img, int x, int y, Color bgcolor, ImageObserver observer) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWIMAGE1, img, x, y, bgcolor, observer) ); 
    } 

    void drawImage(Image img, int x, int y, ImageObserver observer) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWIMAGE2, img, x, y, observer) ); 
    } 

    void drawImage(Image img, int x, int y, int width, int height, Color bgcolor, ImageObserver observer) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWIMAGE3, img, x, y, width, height, bgcolor, observer) ); 
    } 

    void drawImage(Image img, int x, int y, int width, int height, ImageObserver observer) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWIMAGE4, img, x, y, width, height, observer) ); 
    } 

    /** 
     * Disegna una linea tra i punti (x1,y1) e (x2,y2), 
     * usando il colore corrente. 
     */ 
    public void drawLine(int x1, int y1, int x2, int y2) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWLINE, x1, y1, x2, y2) ); 
    } 
    
    /** 
     * Disegna un ovale, usando il colore corrente. 
     * L'ovale disegnato  un cerchio o un ellisse 
     * entro il rettangolo di lati sinistro e destro 
     * rispettivamente di coordinate 
     * <code>x</code> e <code>x+width</code>, 
     * e di lati superiore e inferiore di coordinate 
     * rispettivamente <code>y</code> e <code>y+height</code>.
     * 
     * L'ovale copre un'area di larghezza <code>width+1</code> 
     * e altezza <code>height+1</code>. 
     */ 
    public void drawOval(int x, int y, int width, int height) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWOVAL, x, y, width, height) ); 
    } 

    void drawPolygon(int[] xPoints, int[] yPoints, int nPoints) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWPOLYGON1, xPoints, yPoints, nPoints) ); 
    } 

    void drawPolygon(Polygon p) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWPOLYGON2, p) ); 
    } 

    void drawPolyline(int[] xPoints, int[] yPoints, int nPoints) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWPOLYLINE, xPoints, yPoints, nPoints) ); 
    } 

    /** 
     * Disegna un rettangolo, usando il colore corrente. 
     * I lati sinistro e destro del rettangolo hanno coordinata 
     * rispettivamente <code>x</code> e <code>x+width</code>.
     * I lati superiore e inferiore del rettangolo hanno coordinata 
     * rispettivamente <code>y</code> e <code>y+height</code>.
     */ 
    public void drawRect(int x, int y, int width, int height) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWRECT, x, y, width, height) ); 
    } 

    /** 
     * Disegna un rettangolo con gli angoli arrotondati, 
     * usando il colore corrente. 
     * 
     * I lati sinistro e destro del rettangolo hanno coordinata 
     * rispettivamente <code>x</code> e <code>x+width</code>.
     * I lati superiore e inferiore del rettangolo hanno coordinata 
     * rispettivamente <code>y</code> e <code>y+height</code>.
     * Il diametro orizzontale e verticale degli archi degli angoli 
     * sono <code>arcWidth</code> e <code>arcHeight</code>, 
     * rispettivamente. 
     */ 
    public void drawRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWROUNDRECT, x, y, width, height, arcWidth, arcHeight) ); 
    } 

    /** 
     * Disegna la stringa <code>s</code>, usando il colore corrente. 
     * La linea di base del primo carattere della stringa 
     *  il punto di posizione (x,y). 
     */ 
    public void drawString(String s, int x, int y) { 
        addEntry( new DrawingEntry(DrawingEntry.DRAWSTRING2, s, x, y) ); 
    } 

    void fill3DRect(int x, int y, int width, int height, boolean raised) { 
        addEntry( new DrawingEntry(DrawingEntry.FILL3DRECT, x, y, width, height, raised) ); 
    } 

    /** 
     * Disegna (a colore pieno) 
     * un arco circolare o ellittico che copre 
     * il rettangolo caratterizzato dai parametri 
     * <code>x</code>, <code>y</code>, 
     * <code>width</code> e <code>height</code>.
     */ 
    public void fillArc(int x, int y, int width, int height, int startAngle, int arcAngle) { 
        addEntry( new DrawingEntry(DrawingEntry.FILLARC, x, y, width, height, startAngle, arcAngle) ); 
    } 

    /** 
     * Disegna (a colore pieno) un ovale, usando il colore corrente. 
     * L'ovale disegnato  un cerchio o un ellisse 
     * entro il rettangolo di lati sinistro e destro 
     * rispettivamente di coordinate 
     * <code>x</code> e <code>x+width</code>, 
     * e di lati superiore e inferiore di coordinate 
     * rispettivamente <code>y</code> e <code>y+height</code>.
     */ 
    public void fillOval(int x, int y, int width, int height) { 
        addEntry( new DrawingEntry(DrawingEntry.FILLOVAL, x, y, width, height) ); 
    } 

    void fillPolygon(int[] xPoints, int[] yPoints, int nPoints) { 
        addEntry( new DrawingEntry(DrawingEntry.FILLPOLYGON1, xPoints, yPoints, nPoints) ); 
    } 

    void fillPolygon(Polygon p) { 
        addEntry( new DrawingEntry(DrawingEntry.FILLPOLYGON2, p) ); 
    } 

    /** 
     * Disegna (a colore pien) un rettangolo, usando il colore corrente. 
     * I lati sinistro e destro del rettangolo hanno coordinata 
     * rispettivamente <code>x</code> e <code>x+width</code>.
     * I lati superiore e inferiore del rettangolo hanno coordinata 
     * rispettivamente <code>y</code> e <code>y+height</code>.
     */ 
    public void fillRect(int x, int y, int width, int height) { 
        addEntry( new DrawingEntry(DrawingEntry.FILLRECT, x, y, width, height) ); 
    } 

    /** 
     * Disegna (a colore pieno) un rettangolo con gli angoli arrotondati, 
     * usando il colore corrente. 
     * 
     * I lati sinistro e destro del rettangolo hanno coordinata 
     * rispettivamente <code>x</code> e <code>x+width</code>.
     * I lati superiore e inferiore del rettangolo hanno coordinata 
     * rispettivamente <code>y</code> e <code>y+height</code>.
     * Il diametro orizzontale e verticale degli archi degli angoli 
     * sono <code>arcWidth</code> e <code>arcHeight</code>, 
     * rispettivamente. 
     */ 
    public void fillRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) { 
        addEntry( new DrawingEntry(DrawingEntry.FILLROUNDRECT, x, y, width, height, arcWidth, arcHeight) ); 
    } 

    /** 
     * Rende <code>c</code> il colore corrente. 
     * Tutte le successive operazioni grafiche utilizzeranno 
     * il colore specificato. 
     */ 
    public void setColor(Color c) { 
        addEntry( new DrawingEntry(DrawingEntry.SETCOLOR, c) ); 
    }

    /** 
     * Transla l'origine dei punti nel punto di coordinate 
     * <code>(x,y)</code> riferito al sistema corrente di coordinate. 
     * 
     * Modifica il sistema di coordinate in modo tale che 
     * la nuova origine sia nel punto <code>(x,y)</code> 
     * del sistema di coordinate corrente. 
     * Tutte le coordinate specificate nelle successive 
     * operazioni grafiche saranno considerate relativamente 
     * al nuovo sistema di coordinate. 
     */ 
    public void translate(int x, int y) { 
        addEntry( new DrawingEntry(DrawingEntry.TRANSLATE, x, y) ); 
    } 

    /** 
     * Aggiunge al pannello l'entry grafica e. 
     */ 
    void addEntry(DrawingEntry e) { 
        panel.add(e);  
    } 
} 