/*
 * @(#)SimpleReader.java    3.01 2001/02/28
 *
 * Luca Cabibbo 
 * Dipartimento di Informatica e Automazione 
 * Universit degli Studi Roma Tre 
 * cabibbo@dia.uniroma3.it 
 */ 

package fiji.io; 

import java.io.*; 

/** 
 * Un oggetto <code>SimpleStream</code> permette di leggere, in modo 
 * semplice, dati da un flusso di ingresso. 
 * In particolare, un oggetto <code>SimpleStream</code> consente 
 * la lettura di dati (in formato testuale) corrispondenti 
 * ai vari tipi primitivi di Java. 
 * Sono inoltre fornite funzionalit che permettono di verificare 
 * se la scansione  giunta alla fina della riga o del flusso 
 * di ingresso. </p> 
 *
 * <p> Solitamente, un <code>SimpleStream</code> viene creato senza 
 * specificare argomenti, e questo corrisponde a fargli accedere i 
 * dati dal flusso standard di ingresso <code>System.in</code> 
 * (cio, dalla tastiera). 
 * E' possibile anche creare un <code>SimpleStream</code> che accede 
 * i dati di un altro flusso di ingresso pre-esistente, 
 * ad esempio corrispondente a una stringa, un file di testo 
 * o al contenuto di una pagina Web. 
 * (Si veda in proposito la documentazione associata ai costruttori.)
 * </p> 
 * 
 * @version 3.01 
 * @author Luca Cabibbo 
 */ 
public class SimpleReader extends FilterReader { 
    
    /* Il prossimo carattere da scandire, oppure 0. */ 
    private char next; 

    /** 
     * Crea un <code>SimpleReader</code> corrispondente al 
     * flusso standard di ingresso, <code>System.in</code> 
     * (cio la tastiera). 
     */ 
    public SimpleReader() { 
        this(new InputStreamReader(System.in)); 
    } 
    
    /** 
     * Crea un <code>SimpleReader</code> per leggere 
     * dalla stringa <code>s</code>. 
     * 
     * <p> Esempio: Lettura e somma di due numeri interi: </p> 
     * 
     * <pre>
     *     SimpleReader in = new SimpleReader("2 3"); 
     *     int a, b, somma; 
     *     a = in.readInt(); 
     *     b = in.readInt(); 
     *     somma = a+b; 
     * </pre> 
     * 
     * <p> La stringa <code>s</code> viene eventualmente 
     * estesa con un terminatore di linea. 
     * (Ovvero, viene garantito che la stringa da cui verr 
     * effettuata la lettura termini con un terminatore di linea. 
     * Questo risolve alcuni problemi riscontrati in pratica.) </p> 
     */ 
    public SimpleReader(String s) { 
        this(new StringReader(termina(s))); 
    } 
    
    /* Termina s con un terminatore di linea, 
     * se s non  gi terminata in questo modo. */ 
    private static String termina(String s)  { 
        if (s!=null)  { 
            int lunghezza = s.length(); 
            /* se s non termina con un terminatore di linea */ 
            if (lunghezza>0 && s.charAt(lunghezza-1)!='\n') 
                /* estende s con un terminatore di linea */ 
                s += '\n'; 
        } 
        return s; 
    } 
    
    /** 
     * Crea un <code>SimpleReader</code> corrispondente al 
     * flusso ingresso pre-esistente <code>in</code>. 
     * 
     * <p> Il flusso di ingresso pre-esistente pu essere, 
     * ad esempio, un file di testo oppure una pagina Web. 
     * Nei due casi in esempio, si possono usare rispettivamente 
     * le seguenti seguenze di istruzioni: </p> 
     * 
     * <p> Lettura da un file di testo: </p> 
     * 
     * <pre>
     * // import java.io.*; 
     * // legge dal file "nomefile" 
     * try { 
     *     SimpleReader fileIn = 
     *             new SimpleReader(new FileReader("nomefile")); 
     *     ... scansione del file di testo ... 
     * } catch (FileNotFoundException e) { 
     *     e.printStackTrace(System.out); 
     *     System.exit(-1); 
     * } 
     * </pre> 
     * 
     * <p> Lettura da una pagina Web: </p> 
     * 
     * <pre>
     * // import java.io.*; 
     * // import java.net.*; 
     * // legge dalla pagina di URL "url" 
     * try { 
     *     URL u = new URL("url"); 
     *     URLConnection uc = u.openConnection(); 
     *     InputStream is = uc.getInputStream(); 
     *     Reader r = new BufferedReader(new InputStreamReader(is)); 
     *     SimpleReader netIn = new SimpleReader(r); 
     *     ... scansione della pagina Web ... 
     * } catch (IOException e) { 
     *     e.printStackTrace(System.out); 
     *     System.exit(-1); 
     * } 
     * </pre> 
     * 
     * @param in    Il flusso di ingresso pre-esistente.
     */ 
    public SimpleReader(Reader in) { 
        super(in); 
        this.next = (char) 0; 
    } 
      
    /** 
     * Legge una parola senza spazi al suo interno. 
     * 
     * <p> Ignorando gli eventuali spazi bianchi iniziali, 
     * legge una parola composta da una sequenza non vuota di caratteri 
     * che con comprende spazi bianchi e la restituisce come stringa. 
     * Una parola  quindi una sequenza di caratteri stampabili (non bianchi), 
     * delimitata da spazi e/o dalla fine della linea (o del flusso). 
     * Tutti i caratteri che precedono il primo carattere stampabile 
     * sono considerati spazi bianchi, e quindi ignorati 
     * perch considerati non appartenenti alla parola. 
     * La parola si considera terminata dal primo spazio bianco o 
     * terminazione di linea (o del flusso) che segue il primo 
     * carattere stampabile. 
     * Il carattere che termina la parola 
     * (spazio bianco o carattere di fine linea)
     * viene considerato scandito. </p> 
     *
     * <p> Se si vuole leggere una stringa disposta su una 
     * linea di testo, restituendo anche eventuali spazi iniziali, interi 
     * e finali della linea di testo,  necessario usare il metodo 
     * <code>String readLine()</code>. </p>
     * 
     * @return    La stringa letta. 
     */ 
    public String readString() { 
        StringBuffer string = new StringBuffer(); 
        char ch; 
        skipWhite(); 
        ch = get(); 
        while (!isWhite(ch)) { 
            string.append(ch); 
            ch = get(); 
        } 
        return string.toString();
    } 

    /** 
     * Legge un singolo carattere. 
     * 
     * <p> Gli eventuali caratteri di fine linea vengono ignorati. 
     * </p> 
     *
     * @return    Il carattere letto. 
     */ 
    public char readChar() { 
        char ch; 
        while (eoln()) 
            skipEOLN(); 
        return get(); 
    } 

    /** 
     * Legge un intero (delimitato da spazi). 
     * 
     * <p> Il numerale deve essere espresso in base 10, 
     * e pu iniziare con un segno negativo. 
     * Tutti i caratteri che precedono il primo carattere stampabile 
     * sono considerati spazi bianchi, e non appartengono 
     * al numerale. 
     * Il numerale si considera terminato dal primo spazio bianco o 
     * terminazione di linea (o del flusso) che segue il primo 
     * carattere stampabile del numerale. 
     * Il carattere che termina il numerale 
     * (spazio bianco o carattere di fine linea)
     * si considera scandito. </p> 
     *
     * @return    Il valore <code>long</code> letto. 
     */ 
    public long readLong() { 
        /* assume che la prossima stringa sia l'intero da leggere */ 
        String string = readString(); 
        long numero; 
        try { 
            numero = Long.parseLong(string); 
        } catch (NumberFormatException e) { 
            numero = 0; 
        } 
        return numero;
    } 

    /** 
     * Legge un intero (delimitato da spazi). 
     * 
     * <p> Il numerale deve essere espresso in base 10, 
     * e pu iniziare con un segno negativo. 
     * Tutti i caratteri che precedono il primo carattere stampabile 
     * sono considerati spazi bianchi, e non appartengono 
     * al numerale. 
     * Il numerale si considera terminato dal primo spazio bianco o 
     * terminazione di linea (o del flusso) che segue il primo 
     * carattere stampabile del numerale. 
     * Il carattere che termina il numerale 
     * (spazio bianco o carattere di fine linea)
     * si considera scandito. </p> 
     *
     * @return    Il valore <code>int</code> letto. 
     */ 
    public int readInt() { 
        return (int) readLong(); 
    } 

    /** 
     * Legge un intero (delimitato da spazi). 
     * 
     * <p> Il numerale deve essere espresso in base 10, 
     * e pu iniziare con un segno negativo. 
     * Tutti i caratteri che precedono il primo carattere stampabile 
     * sono considerati spazi bianchi, e non appartengono 
     * al numerale. 
     * Il numerale si considera terminato dal primo spazio bianco o 
     * terminazione di linea (o del flusso) che segue il primo 
     * carattere stampabile del numerale. 
     * Il carattere che termina il numerale 
     * (spazio bianco o carattere di fine linea)
     * si considera scandito. </p> 
     *
     * @return    Il valore <code>short</code> letto. 
     */ 
    public short readShort() { 
        return (short) readLong(); 
    } 

    /** 
     * Legge un numero razionale (delimitato da spazi). 
     * 
     * <p> Il numerale deve essere espresso in base 10, 
     * e pu iniziare con un segno negativo. 
     * Tutti i caratteri che precedono il primo carattere stampabile 
     * sono considerati spazi bianchi, e non appartengono 
     * al numerale. 
     * Il numerale si considera terminato dal primo spazio bianco o 
     * terminazione di linea (o del flusso) che segue il primo 
     * carattere stampabile del numerale. 
     * Il carattere che termina il numerale 
     * (spazio bianco o carattere di fine linea)
     * si considera scandito. </p> 
     *
     * @return    Il valore <code>float</code> letto. 
     */ 
    public float readFloat() { 
        /* assume che la prossima stringa sia il float da leggere */ 
        String string = readString(); 
        float numero; 
        try { 
            numero = (new Float(string)).floatValue(); 
        } catch (NumberFormatException e) { 
            numero = 0; 
        } 
        return numero;
    } 

    /** 
     * Legge un numero razionale (delimitato da spazi). 
     * 
     * <p> Il numerale deve essere espresso in base 10, 
     * e pu iniziare con un segno negativo. 
     * Tutti i caratteri che precedono il primo carattere stampabile 
     * sono considerati spazi bianchi, e non appartengono 
     * al numerale. 
     * Il numerale si considera terminato dal primo spazio bianco o 
     * terminazione di linea (o del flusso) che segue il primo 
     * carattere stampabile del numerale. 
     * Il carattere che termina il numerale 
     * (spazio bianco o carattere di fine linea)
     * si considera scandito. </p> 
     *
     * @return    Il valore <code>double</code> letto. 
     */ 
    public double readDouble() { 
        /* assume che la prossima stringa sia il double da leggere */ 
        String string = readString(); 
        double numero; 
        try { 
            numero = (new Double(string)).doubleValue(); 
        } catch (NumberFormatException e) { 
            numero = 0; 
        } 
        return numero;
    } 

    /** 
     * Legge un <code>boolean</code> (delimitato da spazi). 
     * 
     * <p> I letterali <code>boolean</code> sono della 
     * forma <code>true</code> oppure <code>false</code>. 
     * Altri letterali sono interpretati come <code>false</code>. 
     * </p> 
     *  
     * <p> Tutti i caratteri che precedono il primo carattere 
     * stampabile sono considerati spazi bianchi, e non appartengono 
     * al letterale. 
     * Il letterale si considera terminato dal primo spazio bianco o 
     * terminazione di linea (o del flusso) che segue il primo 
     * carattere stampabile del numerale. 
     * Il carattere che termina il letterale 
     * (spazio bianco o carattere di fine linea)
     * si considera scandito. </p> 
     *
     * @return    Il valore <code>boolean</code> letto. 
     */ 
    public boolean readBoolean() { 
        /* assume che la prossima stringa sia il boolean da leggere */ 
        String string = readString(); 
        boolean b; 
        try { 
            b = (new Boolean(string)).booleanValue(); 
        } catch (NumberFormatException e) { 
            b = false; 
        } 
        return b;
    } 
     
    /** 
     * Legge una linea di testo. 
     * 
     * <p> La stringa restituita non comprende i caratteri di 
     * terminazione della linea. </p> 
     *
     * @return    La porzione della linea corrente 
     *            non ancora scandita. 
     */ 
    public String readLine() { 
        StringBuffer line = new StringBuffer(); 
        char ch; 
        while (!eoln()) { 
            ch = get(); 
            line.append(ch); 
        } 
        skipEOLN(); 
        return line.toString();
    } 

    /** 
     * Consuma il resto della linea corrente, compresi i 
     * caratteri di terminazione di linea. 
     */ 
    public void readln() { 
        char ch; 
        while (!eoln()) { 
            ch = get(); 
        } 
        skipEOLN(); 
    } 
    
    /** 
     * Verifica se  stata raggiunta la fine del flusso. 
     *
     * @return    <code>true</code>  se  stata raggiunta la fine 
     *                               del flusso, 
     *            <code>false</code> altrimenti.  
     */ 
    public boolean eof() { 
        char ch = peek(); 
        return (ch==(char)-1); 
    } 
     
    /** 
     * Verifica se  stata raggiunta la fine della linea corrente. 
     *
     * <p> La linea si considera terminata quando viene scandita una 
     * delle sequenti sequenze di caratteri: 
     * '\r', '\n', oppure '\r' seguito da '\n', 
     * oppure quando viene raggiunta la fine del flusso. </p> 
     *
     * @return    <code>true</code>  se  stata raggiunta la fine 
     *                               della linea corrente, 
     *            <code>false</code> altrimenti.  
     */ 
    public boolean eoln() { 
        char ch = peek(); 
        return (ch=='\n') || (ch=='\r') || eof(); 
    } 
     
    /** Restituisce il carattere corrente, senza consumarlo. */ 
    private char peek() { 
        if (next==0) 
            next = scan(); 
        return next; 
    } 

    /** Consuma il carattere corrente, e lo restituisce. */ 
    private char get() { 
        char ch = peek(); 
        next = 0;  // lo considera comunque consumato 
        return ch; 
    } 

    /** Restituisce un carattere letto da in. */ 
    private char scan() { 
        char ch; 
        try { 
            ch = (char) in.read(); 
        } catch (IOException e) { 
            /* assumiamo di aver raggiunto l'EOF */ 
            ch = (char) -1; 
        } 
        return ch; 
    } 
     
    /** 
     * Consuma gli eventuali caratteri di fine linea: 
     *     in Windows/DOS: \r\n 
     *     in Unix:        \n 
     */ 
    private void skipEOLN() { 
        char ch = peek(); 
        if (ch=='\r') { 
            get(); 
            ch = peek(); 
        } 
        if (ch=='\n') { 
            get(); 
        } 
    } 

    /** 
     * Consuma gli eventuali caratteri bianchi, 
     * cio blank, tabulatori e fine linea. 
     */ 
    private void skipWhite() { 
        char ch = peek(); 
        while (isWhite(ch)) { 
            get(); 
            ch = peek(); 
        } 
    } 

    /** 
     * Verifica se un carattere  assimilabile a uno spazio bianco, 
     * cio  un blank, un tabulatori o un carattere di fine linea. 
     */ 
    private static boolean isWhite(char ch) { 
        return Character.isWhitespace(ch); 
    } 

    /** Verifica se un carattere  assimilabile a una cifra. */ 
    private static boolean isDigit(char ch) { 
        return Character.isDigit(ch); 
    } 

} 