package fiji.robot; 

import java.awt.Color; 
import java.awt.Dimension; 
import java.awt.Font; 
import java.awt.FontMetrics; 
import java.awt.Point; 
import java.awt.Graphics; 

import javax.swing.*; 

/** 
 * canvas per la visualizzazione 
 * di un robot in un labirinto 
 */ 
class LRPanel extends JPanel { 
    
    /* il labirinto */ 
    private Labirinto l; 
    /* l'eventuale robot nel labirinto */
    private Robot r; 

    /* costanti per la visualizzazione del labirinto */ 
    static final int EDGE = 80;   // lato di una cella 
    static final int XOFFSET = 44; // offset per iniziare a disegnare 
    static final int YOFFSET = 40; 
    static final int WOFFSET = 100; // offset per larghezza e altezza nella LRFrame 
    static final int HOFFSET = 100; 
    private static final int FONTSIZE = 32; 
    private static final Color COLORE_SFONDO = Color.lightGray; 
    private static final Color COLORE_BORDO = Color.black; 
    private static final Color COLORE_IN = new Color(204,255,204); // verde chiaro
    private static final Color COLORE_OUT = new Color(204,255,255); // celeste
    private static final Color COLORE_NO = Color.white; 
    
    // private static final String IN = "IN"; 
    static final String OUT = "OUT"; 

    private String IN;   // assegnato dal costruttore 

    /* thread di esecuzione */ 
    // private LRThread t; 
    
    /* costanti per la visualizzazione del robot */ 
    private static final int RADIUS = 32;      // raggio robot 
    private static final int PHI = 16;         // diametro occhi 
    private static final int PUP = 8;          // diametro pupille 
    private static final int ALFA = 3;         // scostamento occhi dal centro 
    private static final int BETA = 5;         // scostamento occhi dal centro 
    private static final int GAMMA = 6;        // scostamento pupille
    private static final int DELTA = PHI-GAMMA-PUP;    // scostamento pupille
    static final Color COLORE_CORPO = Color.red; 
    static final Color COLORE_OCCHI = Color.white; 
    static final Color COLORE_PUPILLE = new Color(51,51,255); // azzurri 
    
    /* crea un nuovo robot in un labirinto, 
     * data la descrizione del tipo del labirinto */ 
    public LRPanel(Labirinto l, Robot r) { 
        super(); 
        this.l = l; 
        this.r = r; 

        /* direzione iniziale del robot */ 
        switch (l.direzioneIniziale()) { 
        case Robot.WEST: 
            IN = "\u2190"; 
            // IN = "\u25c4"; 
            break; 
        case Robot.NORTH: 
            IN = "\u2191"; 
            // IN = "\u25b2"; 
            break; 
        case Robot.EAST: 
            IN = "\u2192"; 
            // IN = "\u25ba"; 
            break; 
        case Robot.SOUTH: 
            IN = "\u2193"; 
            // IN = "\u25bc"; 
            break; 
        } 
        /* non funziona bene */ 
        IN = "IN"; 
        
        /* dimensionamento per la visualizzazione del labirinto */ 
        Dimension dl = l.dimensioni();  // dimensioni del labirinto 
        this.setSize(dl.width*EDGE+2*XOFFSET, dl.height*EDGE+2*YOFFSET); 
        this.setBackground(COLORE_SFONDO);
        this.setVisible(true); 
    
        /* esecuzione concorrente */ 
        // t = new LRThread(this); 
        // t.start();  
    } 
    
    public void setRobot(Robot r) { 
        this.r = r; 
    } 
    
    public Robot getRobot() { 
        return this.r; 
    } 

    /* disegna la frame */ 
    public void paintComponent(Graphics g) { 
        /* ridisegna il componente esteso ... */ 
        super.paintComponent(g); 
        /* ... e disegna le componenti specifiche */
        paintLabirinto(g); 
        if (r!=null) 
            paintRobot(g); 
    } 

    /* disegna il labirinto */     
    public void paintLabirinto(Graphics g) { 
        Dimension dl = l.dimensioni();  // dimensioni del labirinto 
        int height = dl.height; 
        int width = dl.width; 

        /* per ripristinare il font di default */ 
        Font tmp = g.getFont(); 

        Font f = new Font("Dialog", Font.PLAIN, FONTSIZE); 
        g.setFont(f); 
        FontMetrics fm = getFontMetrics(f); 
        int DX_IN = fm.stringWidth(IN)/2; 
        int DX_OUT = fm.stringWidth(OUT)/2; 
        int DY = fm.getAscent()/2; 
        
        for (int y=0; y<height; y++) 
            for (int x=0; x<width; x++) { 
                Point p = new Point(x,y); 
                byte cella = l.tipoCella(p); 
                if (cella!=Labirinto.NO) {   // non  una cella 
                    g.setColor(COLORE_BORDO); 
                    g.drawRect(XOFFSET+x*EDGE, YOFFSET+y*EDGE, EDGE, EDGE); 
                } 
                if (cella==Labirinto.IN) {   // ingresso 
                    g.setColor(COLORE_IN); 
                    g.fillRect(XOFFSET+x*EDGE+1, YOFFSET+y*EDGE+1, 
                               EDGE-1, EDGE-1); 
                    g.setColor(COLORE_BORDO); 
                    g.drawString(IN, XOFFSET+x*EDGE+EDGE/2 - DX_IN, 
                                     YOFFSET+y*EDGE+EDGE/2 + DY); 
                } else if (cella==Labirinto.OUT) { // uscita 
                    g.setColor(COLORE_OUT); 
                    g.fillRect(XOFFSET+x*EDGE+1, YOFFSET+y*EDGE+1, 
                               EDGE-1, EDGE-1); 
                    g.setColor(COLORE_BORDO); 
                    g.drawString(OUT, XOFFSET+x*EDGE+EDGE/2 - DX_OUT, 
                                      YOFFSET+y*EDGE+EDGE/2 + DY); 
                } else if (cella!=Labirinto.NO) {
                    g.setColor(COLORE_NO); 
                    g.fillRect(XOFFSET+x*EDGE+1, YOFFSET+y*EDGE+1, 
                               EDGE-1, EDGE-1); 
                } 

                if ((cella & Labirinto.N) !=0) { // muro a north 
                    g.setColor(COLORE_BORDO); 
                    g.fillRect(XOFFSET+x*EDGE-2, YOFFSET+y*EDGE-2, 
                               EDGE+5, 5);                     
                } 
                if ((cella & Labirinto.S) !=0) { // muro a south
                    g.setColor(COLORE_BORDO); 
                    g.fillRect(XOFFSET+x*EDGE-2, YOFFSET+y*EDGE+EDGE-2, 
                               EDGE+5, 5);                     
                } 
                if ((cella & Labirinto.W) !=0) { // muro a west
                    g.setColor(COLORE_BORDO); 
                    g.fillRect(XOFFSET+x*EDGE-2, YOFFSET+y*EDGE-2, 
                               5, EDGE+5);                     
                } 
                if ((cella & Labirinto.E) !=0) { // muro a east
                    g.setColor(COLORE_BORDO); 
                    g.fillRect(XOFFSET+x*EDGE+EDGE-2, YOFFSET+y*EDGE-2, 
                               5, EDGE+5); 
                } 
            } 
        /* ripristina il font di default */ 
        g.setFont(tmp); 

    } 

    /* disegna il robot */ 
    public void paintRobot(Graphics g) { 
        Dimension dl = l.dimensioni();  // dimensioni del labirinto 
        int height = dl.height; 
        int width = dl.width; 
        
        /* disegna il robot */ 
        int x = r.posizione().x; 
        int y = r.posizione().y; 
        int dir = r.direzione(); 
        g.setColor(COLORE_CORPO); 
        /* corpo del robot */ 
        g.fillOval(XOFFSET+x*EDGE+(EDGE/2-RADIUS), 
                   YOFFSET+y*EDGE+(EDGE/2-RADIUS), 
                   2*RADIUS, 2*RADIUS); 
        /* occhi e pupille del robot */ 
        if (dir==Robot.EAST) { 
            g.setColor(COLORE_OCCHI); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)+ALFA, 
                       YOFFSET+y*EDGE+(EDGE/2)-(BETA+PHI), 
                       PHI, PHI); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)+ALFA, 
                       YOFFSET+y*EDGE+(EDGE/2)+BETA, 
                       PHI, PHI); 
            g.setColor(COLORE_PUPILLE); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)+ALFA+GAMMA, 
                       YOFFSET+y*EDGE+(EDGE/2)-BETA-PHI/2-PUP/2, 
                       PUP, PUP); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)+ALFA+GAMMA, 
                       YOFFSET+y*EDGE+(EDGE/2)+BETA+PHI/2-PUP/2, 
                       PUP, PUP); 
        } else if (dir==Robot.WEST) {
            g.setColor(COLORE_OCCHI); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)-(ALFA+PHI), 
                       YOFFSET+y*EDGE+(EDGE/2)-(BETA+PHI), 
                       RADIUS/2, RADIUS/2); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)-(ALFA+PHI), 
                       YOFFSET+y*EDGE+(EDGE/2)+BETA, 
                       PHI, PHI); 
            g.setColor(COLORE_PUPILLE); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)-ALFA-PHI+DELTA, 
                       YOFFSET+y*EDGE+(EDGE/2)-BETA-PHI/2-PUP/2, 
                       PUP, PUP); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)-ALFA-PHI+DELTA, 
                       YOFFSET+y*EDGE+(EDGE/2)+BETA+PHI/2-PUP/2, 
                       PUP, PUP); 
        } else if (dir==Robot.NORTH) {
            g.setColor(COLORE_OCCHI); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)-(BETA+PHI), 
                       YOFFSET+y*EDGE+(EDGE/2)-(ALFA+PHI), 
                       PHI, PHI); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)+BETA, 
                       YOFFSET+y*EDGE+(EDGE/2)-(ALFA+PHI), 
                       PHI, PHI); 
            g.setColor(COLORE_PUPILLE); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)-BETA-PHI/2-PUP/2, 
                       YOFFSET+y*EDGE+(EDGE/2)-ALFA-PHI+DELTA, 
                       PUP, PUP); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)+BETA+PHI/2-PUP/2, 
                       YOFFSET+y*EDGE+(EDGE/2)-ALFA-PHI+DELTA, 
                       PUP, PUP); 
        } else if (dir==Robot.SOUTH) {
            g.setColor(COLORE_OCCHI); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)-(BETA+PHI), 
                       YOFFSET+y*EDGE+(EDGE/2)+ALFA, 
                       PHI, PHI); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)+BETA, 
                       YOFFSET+y*EDGE+(EDGE/2)+ALFA, 
                       PHI, PHI); 
            g.setColor(COLORE_PUPILLE); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)-BETA-PHI/2-PUP/2, 
                       YOFFSET+y*EDGE+(EDGE/2)+ALFA+GAMMA, 
                       PUP, PUP); 
            g.fillOval(XOFFSET+x*EDGE+(EDGE/2)+BETA+PHI/2-PUP/2, 
                       YOFFSET+y*EDGE+(EDGE/2)+ALFA+GAMMA, 
                       PUP, PUP); 
        }  
    }
    
} 