/*
 * @(#)Labirinto.java    3.0 2001/01/31
 *
 * Luca Cabibbo 
 * Dipartimento di Informatica e Automazione 
 * Universit degli Studi Roma Tre 
 * cabibbo@dia.uniroma3.it 
 */ 

package fiji.robot; 

import java.awt.Dimension; 
import java.awt.Point; 

/** 
 * Un <code>Labirinto</code>  una griglia di celle 
 * quadrate, con un unico ingresso e una unica uscita. 
 * Celle adiacenti possono essere collegate, 
 * oppure separate da un muro. 
 * In un labirinto esiste sempre almeno un percorso 
 * dall'ingresso all'uscita del labirinto. 
 * L'ingresso e l'uscita del labirinto sono considerate 
 * esterne al labirinto. 
 * Nel labirinto pu essere collocato un robot. 
 * 
 * @version 3.0 
 * @author Luca Cabibbo 
 */ 
public class Labirinto { 
    
    /* costanti per la descrizione delle celle del labirinto */ 
    /** Cella di ingresso del labirinto. */ 
    static final byte IN     = 0x10;       // ingresso 
    /** Cella di uscita del labirinto. */ 
    static final byte OUT    = 0x20;       // uscita 
    /** Non  una cella del labirinto. */ 
    static final byte NO     = 0x40;       // chiuso 
        
    /* posizione dei muri */ 
    /** Cella senza muri. */ 
    static final byte FREE   = 0x00;       // nessun muro 
    /** Cella con un muro a nord. */ 
    static final byte N      = 0x01;       // north 
    /** Cella con un muro a est. */ 
    static final byte E      = 0x02;       // east 
    /** Cella con un muro a sud. */ 
    static final byte S      = 0x04;       // south 
    /** Cella con un muro a ovest. */ 
    static final byte W      = 0x08;       // west 
    /** Cella con un muro a nord e uno a est. */ 
    static final byte NE     = N|E;        // north + east 
    /** Cella con un muro a nord e uno a sud. */ 
    static final byte NS     = N|S;        // north + south
    /** Cella con un muro a nord e uno a ovest. */ 
    static final byte NW     = N|W;        // north + west 
    /** Cella con un muro a est e uno a sud. */ 
    static final byte ES     = E|S;        // east + south 
    /** Cella con un muro a est e uno a ovest. */ 
    static final byte EW     = E|W;        // east + west 
    /** Cella con un muro a sud e uno a ovest. */ 
    static final byte SW     = S|W;        // south + west 
    /** Cella con un muro a nord, uno a est e uno a sud. */ 
    static final byte NES    = N|E|S;      // north + east + south 
    /** Cella con un muro a nord, uno a est e uno a ovest. */ 
    static final byte NEW    = N|E|W;      // north + east + west 
    /** Cella con un muro a nord, uno a sud e uno a ovest. */ 
    static final byte NSW    = N|S|W;      // north + south + west 
    /** Cella con un muro a est, uno a sud e uno a ovest. */ 
    static final byte ESW    = E|S|W;      // east + south + west 
    /** Cella circondata da muri in tutte le direzioni. */ 
    static final byte NESW   = N|E|S|W;    // north + east + south + west 
    
    /* Tipologie di labirinti predefiniti. */ 
    /* Labirinto a struttura casuale */ 
    static final int CASUALE = 0; 
    /* Labirinti a struttura semplice */ 
    static final int SEMPLICE = 1; 
    static final int FACILE = 2; 
    /* Labirinti lineari */ 
    static final int LINEARE_UNO = 3; 
    static final int LINEARE_DUE = 4; 
    static final int LINEARE_QUATTRO = 5; 
    /* Labirinti rettangolari */ 
    static final int RETTANGOLARE_A = 6; 
    static final int RETTANGOLARE_B = 7; 
    /* Labirinti generici */ 
    static final int GENERICO = 8; 
    static final int CURVA_DESTRA = 9; 
    static final int CURVA_SINISTRA = 10; 
    static final int DUE_STANZE = 11; 
    static final int DUE_STANZE_B = 12; 
    static final int PRIMO = SEMPLICE;  
    static final int ULTIMO = DUE_STANZE_B;  
    
    /* rappresentazione mediante array del labirinto */ 
    private byte[][] celle = null; 

    /* l'eventuale robot nel labirinto */ 
    private Robot r; 

    /* posizione iniziale del robot nel labirinto */ 
    private Point posizioneIniziale; 
    /* direzione iniziale del robot nel labirinto */ 
    private int direzioneIniziale; 

    /* descrizione del labirinto */ 
    private String descrizione; 
    
    /* frame per la visualizzazione del labirinto */ 
    private LRFrame f; 
    
    /** 
     * Crea un nuovo labirinto data la descrizione 
     * del tipo del labirinto. 
     * 
     * Inizialmente il labirinto non contiene nessun robot. 
     * 
     * La descrizione del tipo del labirinto deve 
     * essere una delle seguenti: 
     * <code>semplice</code>, 
     * <code>facile</code>, 
     * <code>lineare</code>, 
     * <code>rettangolare</code>, 
     * <code>generico</code>,  
     * <code>casuale</code>. 
     * 
     * @param descrizione    La descrizione del labirinto che deve essere creato. 
     * @exception java.lang.Error    Descrizione non prevista. 
     */ 
    public Labirinto(String descrizione) { 
        /* inizializza la struttura del labirinto */ 
        if (descrizione.equals("semplice")) 
            initLabirinto(SEMPLICE); 
        else if (descrizione.equals("facile")) 
            initLabirinto(FACILE); 
        else if (descrizione.equals("lineare")) 
            initLabirinto(casuale(LINEARE_UNO, LINEARE_QUATTRO)); 
        else if (descrizione.equals("rettangolare")) 
            initLabirinto(casuale(RETTANGOLARE_A, RETTANGOLARE_B)); 
        else if (descrizione.equals("generico")) 
            initLabirinto(casuale(GENERICO,DUE_STANZE_B)); 
        else if (descrizione.equals("casuale")) 
            initLabirinto(CASUALE); 
        else 
            throw new Error("Errore: descrizione non prevista."); 
        
        /* inizialmente non c' nessun robot */ 
        this.r = null; 
        
        /* crea la frame per la visualizzazione del labirinto */ 
        f = new LRFrame(this); 
        
        /* introduce un ritardo */ 
        try  { 
            Thread.sleep(1000); 
        } catch (Exception e)  { 
        } 
        
    } 
    
    /** 
     * Crea un nuovo labirinto di tipo casuale. 
     * 
     * Inizialmente il labirinto non contiene nessun robot. 
     */ 
    public Labirinto() { 
        /* crea un labirinto casuale */
        this("casuale");         
    } 

    /* Genera un numero intero casuale tra inizio e fine 
     * (estremi compresi). */ 
    private static int casuale(int inizio, int fine)  { 
        // pre: inizio<=fine 
        return inizio + (int) (Math.random()*(fine-inizio+1)); 
    } 
    
    /** 
     * Inizializza la struttura di un nuovo labirinto data la codifica 
     * del tipo del labirinto. 
     * Inizialmente il labirinto non contiene nessun robot. 
     */ 
    private void initLabirinto(int tipo) { 
        // pre: CASUALE <= tipo <= ULTIMO

        boolean casuale = tipo==CASUALE; 
        
        /* se casuale, sceglie una tipologia a caso */ 
        if (casuale) 
            tipo = casuale(PRIMO, ULTIMO); 
            
        /* inizializza la struttura del labirinto */ 
        switch (tipo)  { 
            case SEMPLICE: 
                celle = new byte[][] { 
                        { IN, N, N, OUT } , 
                        { NO, ESW, ESW, NO } 
                    } ; 
                posizioneIniziale = new Point(0,0); 
                direzioneIniziale = Robot.EAST; 
                this.descrizione = "semplice"; 
                break; 

            case FACILE: 
                celle = new byte[][] { 
                        { IN, N, NES, NO } , 
                        { NO, SW, NS, OUT } 
                    } ; 
                posizioneIniziale = new Point(0,0); 
                direzioneIniziale = Robot.EAST; 
                this.descrizione = "facile"; 
                break; 
            
            case LINEARE_UNO: 
                celle = new byte[][] { 
                        { IN, NS, OUT } 
                    } ; 
                posizioneIniziale = new Point(0,0); 
                direzioneIniziale = Robot.EAST; 
                this.descrizione = "lineare"; 
                break; 
                
            case LINEARE_DUE: 
                celle = new byte[][] { 
                        { IN, NS, NS, OUT } 
                    } ; 
                posizioneIniziale = new Point(0,0); 
                direzioneIniziale = Robot.EAST; 
                this.descrizione = "lineare"; 
                break; 
                
            case LINEARE_QUATTRO: 
                celle = new byte[][] { 
                        { IN, NS, NS, NS, NS, OUT } 
                    } ; 
                posizioneIniziale = new Point(0,0); 
                direzioneIniziale = Robot.EAST; 
                this.descrizione = "lineare"; 
                break; 
                
            case RETTANGOLARE_A: 
                celle = new byte[][] { 
                        { NO, NW, N, NE } ,  
                        { IN, FREE, FREE, E } , 
                        { NO, SW, FREE, ES } , 
                        { NO, NO, OUT, NO } 
                    } ; 
                posizioneIniziale = new Point(0,1); 
                direzioneIniziale = Robot.EAST; 
                this.descrizione = "rettangolare"; 
                break; 

            case RETTANGOLARE_B: 
                celle = new byte[][] { 
                        { NO, NEW } ,  
                        { IN, E } , 
                        { NO, EW } , 
                        { OUT, E } , 
                        { NO, ESW } 
                    } ; 
                posizioneIniziale = new Point(0,1); 
                direzioneIniziale = Robot.EAST; 
                this.descrizione = "rettangolare"; 
                break; 

            case GENERICO: 
                celle = new byte[][] { 
                        { NO, NSW, NE, NW, NS, OUT } ,  
                        { NO, NW, E, SW, NE, NO } , 
                        { IN, ES, SW, NS, ES, NO } 
                    } ; 
                posizioneIniziale = new Point(0,2); 
                direzioneIniziale = Robot.EAST; 
                this.descrizione = "generico"; 
                break; 
                
            case CURVA_DESTRA: 
                celle = new byte[][] { 
                        { IN, NS, NS, NE, NO, NO } ,  
                        { NO, NO, NO, SW, NS, OUT } 
                    } ; 
                posizioneIniziale = new Point(0,0); 
                direzioneIniziale = Robot.EAST; 
                this.descrizione = "generico"; 
                break; 
            
            case CURVA_SINISTRA: 
                celle = new byte[][] { 
                        { NO, NO, NO, NW, NS, OUT } ,  
                        { IN, NS, NS, ES, NO, NO } 
                    } ; 
                posizioneIniziale = new Point(0,1); 
                direzioneIniziale = Robot.EAST; 
                this.descrizione = "generico"; 
                break; 

            case DUE_STANZE: 
                celle = new byte[][] { 
                        { NO, NW, NE, NW, NE, NO } ,  
                        { NO, W,  E,  W,  FREE, OUT } ,  
                        { IN, FREE, E, W, E, NO } ,  
                        { NO, W, FREE, FREE, E, NO } ,  
                        { NO, SW, ES, SW, ES, NO } 
                    } ; 
                posizioneIniziale = new Point(0,2); 
                direzioneIniziale = Robot.EAST; 
                this.descrizione = "generico"; 
                break; 
            
            case DUE_STANZE_B: 
                celle = new byte[][] { 
                        { NO, NW, NE, NW, N, NE } ,  
                        { OUT, FREE,  FREE,  FREE,  FREE, ES } ,  
                        { NO, W, E, W, E, NO } ,  
                        { IN, FREE, FREE, FREE, E, NO } ,  
                        { NO, SW, ES, SW, ES, NO } 
                    } ; 
                posizioneIniziale = new Point(0,3); 
                direzioneIniziale = Robot.EAST; 
                this.descrizione = "generico"; 
                break; 
            
            default: 
                throw new Error("Errore: tipo non previsto."); 
        }
        
        /* se casuale, corregge la descrizione */ 
        if (casuale) 
            this.descrizione = "casuale"; 
    } 

    /** 
     * Pone il robot r nel labirinto. 
     */ 
    void setRobot(Robot r) { 
        this.r = r; 
        f.setRobot(); 
    } 
    
    /** 
     * Restituisce l'eventuale robot presente nel labirinto. 
     */ 
    Robot getRobot() { 
        return r; 
    } 

    /** 
     * Restituisce la posizione iniziale che deve essere 
     * occupata da un robot nel labirinto. 
     */ 
    Point posizioneIniziale() { 
        return posizioneIniziale; 
    } 
    
    /** 
     * Restituisce la direzione iniziale che deve essere 
     * assunta da un robot nel labirinto. 
     */ 
    int direzioneIniziale() { 
        return direzioneIniziale; 
    } 
    
    /** 
     * Verifica se la posizione p  interna al labirinto. 
     */ 
    boolean dentro(Point p) { 
        int j = p.x; 
        int i = p.y; 
        byte cella = celle[i][j]; 
        return (cella!=IN) && (cella!=OUT) && (cella!=NO); 
    } 
    
    /** 
     * Verifica se la posizione p  esterna al labirinto. 
     */ 
    boolean fuori(Point p) { 
        int j = p.x; 
        int i = p.y; 
        byte cella = celle[i][j]; 
        return (cella==IN) || (cella==OUT); 
    } 

    /** 
     * Verifica se il labirinto contiene un muro 
     * nella posizione p e direzione d. 
     */ 
    boolean muro(Point p, int d) { 
        int j = p.x; 
        int i = p.y; 
        byte cella = celle[i][j]; 
        return (d==Robot.EAST && (cella&E)!=0) || 
               (d==Robot.SOUTH && (cella&S)!=0) || 
               (d==Robot.WEST && (cella&W)!=0) || 
               (d==Robot.NORTH && (cella&N)!=0); 
    } 
    
    /** 
     * Restituisce le dimensioni del labirinto. 
     */ 
    Dimension dimensioni() { 
        return new Dimension(celle[0].length, celle.length); 
    } 
    
    /* 
     * Restituisce il tipo della cella del labirinto 
     * di posizione (x,y). 
    public byte tipo(int x, int y) { 
        return celle[y][x]; 
    } 
     */ 
    
    /** 
     * Restituisce il tipo della cella del labirinto 
     * di posizione p. 
     */ 
    byte tipoCella(Point p) { 
        return celle[p.y][p.x]; 
    } 
    
    /** 
     * Restituisce la descrizione del labirinto. 
     */ 
    String descrizione() { 
        return descrizione; 
    } 
    
    /** 
     * Restituisce la frame in cui deve essere 
     * visualizzato il labirinto. 
     */ 
    LRFrame getFrame() { 
        return f; 
    } 


    /** 
     * Rimuove la frame in cui viene 
     * visualizzato il labirinto. 
     */ 
    void dispose() { 
        getFrame().dispose(); 
    } 
    
    /** 
     * Metodo di test. 
     * Crea e visualizza un labirinto del tipo specificato come 
     * argomento nella linea di comando. 
     * 
     * La descrizione del tipo del labirinto deve 
     * essere una delle seguenti: 
     * <code>semplice</code>, 
     * <code>facile</code>, 
     * <code>lineare</code>, 
     * <code>rettangolare</code>, 
     * <code>generico</code>,  
     * <code>casuale</code>. 
     * 
     * Se non viene specificata nessuna descrizione, 
     * viene creato un labirinto casuale. 
     */ 
    static void main(String[] args) { 
        Labirinto l; 
        String d;    // descrizione, fornita dagli argomenti 
        int i; 
        
        /* calcola la descrizione d del labirinto */ 
        if (args.length<1) 
            d = "casuale"; 
        else { 
            d = args[0]; 
            for (i=1; i<args.length; i++) 
                d += " " + args[i]; 
        } 
        /* crea un labirinto di tipo d */ 
        l = new Labirinto(d); 
    } 
    
} 
