/*
 * @(#)Robot.java    3.0 2001/01/31
 *
 * Luca Cabibbo 
 * Dipartimento di Informatica e Automazione 
 * Universit degli Studi Roma Tre 
 * cabibbo@dia.uniroma3.it 
 */ 

package fiji.robot; 

import java.awt.Point; 
import javax.swing.*; 

/** 
 * Un <code>Robot</code>  un automa virtuale 
 * che si muove in un <code>Labirinto</code>. 
 * Il labirinto  una griglia di celle 
 * quadrate, con un unico ingresso e una unica uscita. 
 * Il robot ha la capacit di muoversi da una cella a 
 * una cella adiacente (nella direzione corrente del robot) 
 * e di cambiare direzione, girando a destra o a sinistra. 
 * Il labirinto contiene ovviamente dei muri che impediscono 
 * alcuni dei movimenti al robot: il robot non pu avanzare 
 * se si trova di fronte a un muro. 
 * Il robot pu essere comandato inviandogli messaggi, 
 * oppure in modo interattivo. 
 * 
 * @version 3.0 
 * @author Luca Cabibbo 
 * @see Labirinto
 */ 
public class Robot { 
    
    /* costanti per l'orientamento del robot */ 
    /** Direzione nord. */ 
    static final int NORTH   = 0; 
    /** Direzione est. */ 
    static final int EAST    = 1; 
    /** Direzione sud. */ 
    static final int SOUTH   = 2; 
    /** Direzione ovest. */ 
    static final int WEST    = 3;  
    
    /* ritardo introdotto nell'animazione dopo una mossa del robot */ 
    private static final int DELAY   = 500;  // ms
    private static final int REPAINT_DELAY   = 100;  // ms

    /* propriet del robot nel labirinto */ 
    private Point posizione; 
    private int direzione; 
    private Labirinto l; 
    
    /* modalit di comando del robot */ 
    private boolean interattivo; 
    
    /* conta le mosse effettuate */ 
    private int mosse; 
        
    /** 
     * Crea un Robot nel Labirinto l, 
     * possibilmente comandabile in modo interattivo. 
     * Il Robot  interattivo se il secondo 
     * argomento  <code>true</true>. 
     * 
     * @param l    Il Labirinto in cui deve essere creato il robot. 
     * @param interattivo    Se il robot deve essere interattivo. 
     * @exception RobotException    Labirinto nullo o non vuoto. 
     */ 
    public Robot(Labirinto l, boolean interattivo) throws RobotException { 
        // pre: l!=null && l non contiene gi un Robot 
        
        /* verifica la pre-condizione */ 
        if (l==null) 
            throw new RobotException("Labirinto nullo"); 
        else if (l.getRobot()!=null) 
            throw new RobotException("Labirinto non vuoto"); 
            
        /* inizializza il robot */ 
        this.l = l; 
        this.posizione = l.posizioneIniziale(); 
        this.direzione = l.direzioneIniziale(); 
        this.interattivo = interattivo; 
        this.mosse = 0; 
        
        /* notifica l'esistenza del robot al labirinto */ 
        l.setRobot(this); 
        repaint(); 
    } 
    
    /** 
     * Crea un Robot nel Labirinto l. 
     * 
     * @param l    Il Labirinto in cui deve essere creato il robot. 
     * @exception RobotException    Labirinto nullo o non vuoto. 
     */ 
    public Robot(Labirinto l) throws RobotException { 
        this(l, false); 
    } 
    
    /** 
     * Fa girare il robot a destra (senza avanzare). 
     */     
    public void giraDestra() { 
        direzione = (direzione+1)%4; 
        mosse++; 
        repaint(); 
    } 
    
    /** 
     * Fa girare il robot a sinistra (senza avanzare). 
     */     
    public void giraSinistra() { 
        direzione = (direzione+3)%4; 
        mosse++; 
        repaint(); 
    } 

    /** 
     * Fa avanzare il robot, muovendolo nella cella adiacente 
     * nella sua direzione corrente. 
     * Questa azione pu essere effettuata solo nelle 
     * due seguenti situazioni: 
     * (i) il robot si trova nella posizione iniziale 
     * ed  nella direzione dell'ingresso del labirinto; 
     * (ii) il robot si trova all'interno del labirinto, 
     * non  di fronte a un muro, e non sta avanzando verso 
     * l'ingresso del labirinto (tentando di uscirne). 
     * Viceversa (se il robot prova a muoversi 
     * dalla posizione iniziale non verso l'interno del 
     * labirinto, o se da una posizione interna prova a 
     * muoversi verso un muro, o se prova a muoversi essendo 
     * gi uscito dal labirinto) viene causato un errore. 
     * 
     * @exception RobotException    Mossa non consentita. 
     */     
    public void avanza() throws RobotException { 
        String error = null; 
        /* verifica se la mossa  consentita */ 
        switch (l.tipoCella(posizione)) { 
        case Labirinto.IN: 
            if (direzione!=l.direzioneIniziale()) 
                error = "Il robot deve entrare nel labirinto"; 
            break; 
        case Labirinto.OUT: 
        case Labirinto.NO: 
            error = "Il robot deve fermarsi"; 
            break; 
        default:             // robot dentro al labirinto 
            if (l.muro(posizione, direzione)) 
                error = "Il robot non pu attraversare un muro"; 
            break; 
        } 
        
        if (error!=null) {
            JOptionPane.showMessageDialog(
                null, "avanza(): Operazione non consentita: " + error); 
            l.dispose(); 
            throw new RobotException(error); 
        }
        
        // ok, mossa consentita 
        switch (direzione) { 
        case NORTH: 
            posizione.y--; 
            break; 
        case SOUTH: 
            posizione.y++; 
            break; 
        case EAST: 
            posizione.x++; 
            break; 
        case WEST: 
            posizione.x--; 
            break; 
        } 
        mosse++; 
        repaint(); 
    } 

    /** 
     * Verifica se il robot  all'interno del labirinto.
     * 
     * @return    L'esito della verifica.  
     */     
    public boolean dentro() { 
        return l.dentro(posizione); 
    } 

    /** 
     * Verifica se il robot, nella posizione corrente, 
     * si trova di fronte a un muro del labirinto. 
     * 
     * @return    L'esito della verifica.  
     */     
    public boolean fronteAlMuro() { 
        return l.muro(posizione, direzione); 
    } 
    
    /** 
     * Restituisce la posizione corrente del robot. 
     */     
    Point posizione() { 
        return posizione; 
    } 
    
    /** 
     * Restituisce la direzione corrente del robot. 
     */     
    int direzione() { 
        return direzione; 
    } 
    
    /** 
     * Verifica se il robot  gestito in modalit interattiva. 
     */     
    boolean isInterattivo() { 
        return interattivo; 
    } 

    /** 
     * Restituisce il numero di mosse effettuate. 
     */     
    int mosse() { 
        return mosse; 
    } 
    
    /* 
     * Ridisegna il robot e il suo labirinto, 
     * introducendo un ritardo se il robot  gestito 
     * in modalit non interattiva. 
     */ 
    private void repaint() { 
        l.getFrame().repaint(); 
        if (!interattivo) { 
            try { 
                Thread.sleep(DELAY); 
            } catch (Exception e) { } 
        }
    } 
    
    /** 
     * Metodo di test. 
     * Crea e visualizza un robot interattivo 
     * in un labirinto del tipo specificato come 
     * argomento nella linea di comando. 
     * 
     * La descrizione del tipo del labirinto deve 
     * essere una delle seguenti: 
     * <code>semplice</code>, 
     * <code>facile</code>, 
     * <code>lineare</code>, 
     * <code>rettangolare</code>, 
     * <code>generico</code>,  
     * <code>casuale</code>. 
     * 
     * Se non viene specificata nessuna descrizione, 
     * viene creato un robot in un labirinto casuale. 
     */ 
    static void main(String[] args) { 
        Labirinto l; 
        Robot r; 
        String d;    // descrizione, fornita dagli argomenti 
        int i; 
        
        /* calcola la descrizione d del labirinto */ 
        if (args.length<1) 
            d = "casuale"; 
        else { 
            d = args[0]; 
            for (i=1; i<args.length; i++) 
                d += " " + args[i];             
        } 
        /* crea un robot in un labirinto di tipo d */ 
        l = new Labirinto(d); 
        r = new Robot(l, true); 
    }
            
} 