/*
 * @(#)ArrayList.java    0.1 2000/05/25
 *
 * Luca Cabibbo 
 * Dipartimento di Informatica e Automazione 
 * Universit degli Studi Roma Tre 
 * cabibbo@dia.uniroma3.it 
 */ 

package fiji.util; 
import java.util.Iterator; 

/** 
 * Un <code>ArrayList</code>  una lista rappresentata mediante array. 
 * ( una implementazione parziale dell'interfaccia 
 * <code>java.util.List</code>.)
 * Ogni <code>ArrayList</code> ha una capacit. La capacit  la 
 * lunghezza dell'array usato per memorizzare gli elementi 
 * della lista, e vale almeno la lunghezza della lista. 
 * Quando vengono aggiunti elementi alla lista, la sua 
 * capacit varia automaticamente. 
 * 
 * @version 0.1 
 * @author Luca Cabibbo 
 */ 
public class ArrayList implements List { 
    
    /** 
     * Elementi della lista. 
     * La lunghezza di <code>data</code>  chiamata 
     * la capacit della lista, che  sempre non 
     * inferiore alla lunghezza della lista. 
     */ 
    private Object[] data; 
    
    /** 
     * Lunghezza della lista. 
     * In pratica, una lista di una certa lunghezza 
     * pu essere rappresentata da un array 
     * <code>data</code> pi lungo del necessario; 
     * gli elementi della lista sono quelli che occupano 
     * le prime <code>size</code> posizioni di 
     * <code>data</code>. 
     */ 
    private int size; 
     
    /** 
     * Crea un nuovo <code>ArrayList</code>, inizialmente vuoto 
     * e di capacit 10. 
     */ 
    public ArrayList() { 
        this(10); 
    } 
    
    /** 
     * Crea un nuovo <code>ArrayList</code>, inizialmente vuoto 
     * e di capacit capacitIniziale. 
     */ 
    public ArrayList(int capacitIniziale) { 
        // pre: capacitIniziale>0
        this.data = new Object[capacitIniziale]; 
        this.size = 0; 
    } 
     
    /** Calcola la lunghezza di questa lista. */ 
    public int size() { 
        return this.size; 
    } 

    /** Verifica se questa lista  vuota. */ 
    public boolean isEmpty() { 
        /* la lista  vuota se ha lunghezza zero */ 
        return this.size==0; 
    } 

    /** 
     * Verifica se questa lista contiene almeno un 
     * elemento uguale a <code>obj</code>. 
     * Pi precisamente, se <code>obj</code>  <code>null</code>, 
     * verifica se questa lista contiene 
     * un elemento <code>null</code>, 
     * altrimenti verifica se contiene un elemento 
     * uguale a <code>obj</code>. 
     */ 
    public boolean contains(Object obj) { 
        return this.indexOf(obj)>=0; 
    } 

    /** 
     * Inserisce <code>obj</code> come ultimo elemento 
     * di questa lista (opzionale). 
     * Restituisce <code>true</code>, perch la lista viene 
     * sicuramente modificata dall'inserimento. 
     */ 
    public boolean add(Object obj) {
        /* inserisce obj in posizione size */  
        this.add(this.size, obj); 
        return true; 
    } 
    
    /** 
     * Rimuove da questa lista il primo elemento 
     * uguale a <code>obj</code>, se presente (opzionale). 
     * Restituisce <code>true</code> se un elemento uguale 
     * a <code>obj</code> 
     *  stato effettivamente rimosso dalla lista. 
     */ 
    public boolean remove(Object obj) {
        int index;    // indice dell'elemento da rimuovere 

        /* cerca obj nella lista */ 
        index = this.indexOf(obj); 
        /* se presente, lo rimuove */ 
        if (index>=0) 
            this.remove(index); 
        return index>=0; 
    } 
    
    /** 
     * Restituisce un iteratore sugli elementi di questa 
     * lista, dal primo verso l'ultimo. 
     */ 
    public Iterator iterator() { 
        /* restituisce un iteratore su questa lista */ 
        return new Itr(); 
    } 
    
    /** 
     * Verifica se questa lista contiene tutti gli 
     * elementi della collezione <code>c</code>. 
     */ 
    public boolean containsAll(Collection c) { 
        // pre: c!=null 
        Iterator i;             // per la scansione di c 
        boolean contieneTutti;  // questa lista contiene 
                                // tutti gli elementi di c 
        
        /* verifica se ogni elemento di c  contenuto 
         * nella lista */ 
        contieneTutti = true; 
        i = c.iterator(); 
        while (contieneTutti && i.hasNext()) 
            if (!this.contains(i.next())) 
                contieneTutti = false; 

        return contieneTutti; 
    } 

    /** 
     * Inserisce tutti gli elementi di <code>c</code> in coda 
     * a questa lista, nell'ordine in cui gli elementi
     * vengono visitati dall'iteratore di <code>c</code> (opzionale).
     * Restituisce <code>true</code> se questa lista viene modificata 
     * dall'inserimento. */ 
    public boolean addAll(Collection c) { 
        // pre: c!=null 
        Iterator i;          // per la scansione di c 
        boolean modificata;  // questa lista 
                             //  stata modificata 

        /* itera su c, inserendo gli elementi 
         * in questa lista */ 
        i = c.iterator(); 
        modificata = false;   // finora nessun inserimento 
        while (i.hasNext()) 
            modificata = this.add(i.next()) || modificata; 

        return modificata; 
    }
    
    /** 
     * Rimuove da questa lista tutti gli elementi 
     * della collezione <code>c</code> (opzionale). 
     * Restituisce <code>true</code> se questa lista viene modificata 
     * dalla rimozione. */ 
    public boolean removeAll(Collection c) { 
        // pre: c!=null 
        Iterator i;          // per la scansione di c 
        boolean modificata;  // la lista  stata modificata 
        
        /* itera su c rimuovendo gli elementi dalla lista */ 
        i = c.iterator(); 
        modificata = false;   // finora nessuna rimozione 
        while (i.hasNext()) 
            modificata = this.remove(i.next()) ||
                         modificata; 
                         
        return modificata; 
    }
    
    /** 
     * Rimuove da questa lista tutti gli elementi che non 
     * sono contenuti nella collezione <code>c</code> (opzionale).
     * Restituisce <code>true</code> se questa lista viene modificata 
     * dalla rimozione. 
     */ 
    public boolean retainAll(Collection c) { 
        // pre: c!=null 
        Iterator i;          // per la scansione della lista
        boolean modificata;  // questa lista 
                             //  stata modificata 

        /* itera su questa lista, rimuovendo gli elementi 
         * che non sono contenuti in c */ 
        i = this.iterator(); 
        modificata = false;   // finora nessuna rimozione 
        while (i.hasNext()) 
            /* se c non contiene l'elemento corrente */ 
            if (!c.contains(i.next())) { 
                /* rimuovilo, usando l'iteratore */ 
                i.remove(); 
                modificata = true; 
            } 

        return modificata; 
    }
    
    /** 
     * Rimuove tutti gli elementi da questa lista 
     * (opzionale). 
     */ 
    public void clear() { 
        int i;    // per la scansione di data 
        
        /* poni a null gli elementi attualmente 
         * significativi di data, per evitare
         * la presenza di oggetti appesi */ 
        for (i=0; i<size; i++) 
            data[i] = null; 
        /* azzera il numero di elementi significativi 
         * di data */ 
        this.size = 0; 
    } 
    
    /** 
     * Restituisce un array che contiene gli elementi di 
     * questa lista, ciascuno nella posizione in cui compare 
     * nella lista. 
     */ 
    public Object[] toArray() { 
        Object[] result;    // array da creare e restituire 
        int i;         // per la scansione di data e result 
        
        result = new Object[this.size]; 
        for (i=0; i<this.size; i++) 
            result[i] = data[i]; 
            
        return result; 
    } 
    
    /** 
     * verifica se questa lista  uguale a <code>obj</code>. 
     * Pi precisamente, restituisce <code>true</code> 
     * se anche <code>obj</code> 
     *  una lista e le due liste sono uguali, 
     * ovvero hanno la stessa lunghezza e ogni elemento 
     * di questa lista  uguale a quello che occupa la 
     * stessa posizione in <code>obj</code>. 
     */ 
    public boolean equals(Object obj) { 
        boolean uguali;   // questa lista e obj sono uguali 
        Iterator i1, i2;  // per la scansione di questa
                          // lista e di obj (se lista) 
        Object o1, o2;    // elementi di questa lista 
                          // e di obj (se lista) 

        if (this==obj)     // se sono lo stesso oggetto 
            uguali = true;     // allora sono uguali 
        else if (obj instanceof List) { 
                           // se anche obj  una lista 
            /* anche obj  una lista */ 
            i1 = iterator();   // per scandire questa lista 
            i2 = ((List) obj).iterator(); //per scandire obj
            uguali = true; //sono uguali se non sono diverse 
            while (i1.hasNext() && i2.hasNext()) { 
                o1 = i1.next(); 
                o2 = i2.next(); 
                // se o1 non  uguale a o2 
                if (!(o1==null && o2==null) && 
                    !(o1!=null && o1.equals(o2))) 
                   uguali = false; 
            } 
            /* questa lista e obj sono uguali se sono state 
             * entrambe scandite completamente senza trovare 
             * differenze */ 
            uguali = uguali && 
                     !i1.hasNext() && !i2.hasNext(); 
        } else   // obj non  una lista 
            uguali = false; 

        return uguali; 
    } 
    
    /** 
     * Restituisce l'elemento di questa lista 
     * di posizione <code>index</code>. 
     */ 
    public Object get(int index) { 
        // pre: index>=0 && index<this.size 
        return data[index]; 
    } 
    
    /** 
     * Sostituisce l'elemento di questa lista 
     * di posizione <code>index</code> con <code>obj</code>. 
     * Restituisce l'elemento che precedentemente 
     * occupava la posizione <code>index</code>. 
     */ 
    public Object set(int index, Object obj) { 
        // pre: index>=0 && index<this.size 
        Object oldObj;    // elemento in posizione index 

        oldObj = data[index]; 
        data[index] = obj; 
        return oldObj; 
    } 

    /** 
     * Inserisce in questa lista l'elemento <code>obj</code> 
     * nella posizione <code>index</code>. 
     * Tutti gli elementi che precedentemente occupavano 
     * una posizione maggiore o uguale a <code>index</code> 
     * vengono spostati di una posizione verso destra. 
     */ 
    public void add(int index, Object obj) { 
        // pre: index>=0 && index<=this.size 
        int lun;           // lunghezza corrente della lista 
        int capacit;      // capacit corrente della lista 
        Object[] oldData;  // riferimento a data corrente 
        int i;             // per la scansione di data 

        lun = this.size;          // lunghezza corrente 
        capacit = data.length;   // capacit corrente 

        /* memorizza il riferimento a data corrente */ 
        oldData = data; 

        /* se necessario, rimpiazza l'array data */ 
        if (lun>=capacit) { 
            /* per effettuare l'inserimento bisogna 
             * sostituire data: lo sostituisco con 
             * un array di capacit doppia */ 
            data = new Object[capacit*2]; 
            /* trascrivo in data i primi index elementi */ 
            for (i=0; i<index; i++) 
                data[i] = oldData[i]; 
        } 
        /* ora i primi index elementi di data sono giusti */ 
        
        /* sposta verso destra gli elementi di posizione 
         * maggiore o uguale a index 
         * (bisogna procedere da destra verso sinistra) */ 
        for (i=lun; i>index; i--) 
            data[i] = oldData[i-1]; 

        /* inserisci obj */ 
        data[index] = obj; 

        /* aggiorna la lunghezza della lista */ 
        this.size++; 
    } 

    /** 
     * Rimuove da questa lista l'elemento di posizione 
     * <code>index</code> e lo restituisce. 
     * Tutti gli elementi che precedentemente occupavano 
     * una posizione maggiore di <code>index</code> vengono 
     * spostati di una posizione verso sinistra. 
     */ 
    public Object remove(int index) { 
        // pre: index>=0 && index<this.size 
        Object obj;     // elemento in posizione index 
        int lun;        // lunghezza corrente della lista 
        int i;          // per la scansione di data 
  
        lun = this.size; 
        obj = data[index];   // elemento da rimuovere 

        /* sposta verso sinistra gli elementi di 
         * posizione maggiore di index  
         * (bisogna procedere da sinistra verso destra) */ 
        for (i=index+1; i<lun; i++) 
            data[i-1] = data[i]; 

        /* poni a null l'elemento di indice size-1, 
         * per evitare la presenza di oggetti appesi */ 
        data[size-1] = null; 

         /* aggiorna la lunghezza della lista */ 
        this.size--; 

        /* restituisci l'elemento rimosso */ 
        return obj; 
    } 
    
    /** 
     * Inserisce in questa lista tutti gli elementi 
     * contenuti nella collezione <code>c</code>, 
     * a partire dalla posizione <code>index</code> (opzionale). 
     * L'inserimento avviene nell'ordine in cui gli elementi 
     * di <code>c</code> vengono visitati dal suo iteratore. 
     * Restituisce <code>true</code> questa lista 
     *  stata modificata dall'inserimento. 
     */ 
    public boolean addAll(int index, Collection c) { 
        // pre: index>=0 && index<=this.size() 
        Iterator i;           // per la scansione di c; 
        boolean modificata;   // questa lista  stata modificata 
        
        /* inserisce tutti gli elementi di c in coda alla lista, 
         * come restituiti dall'iteratore di c */ 
        modificata = false;  
        i = c.iterator(); 
        while (i.hasNext()) { 
            this.add(index, i.next()); 
            modificata = true; 
            index++;  // il prossimo sar inserito alla destra 
                      // di quest'ultimo inserito 
        } 
                
        return modificata; 
    } 
       
    /** 
     * Calcola lindice del primo elemento di questa lista
     * uguale a <code>obj</code>, oppure <code>-1</code> 
     * se la lista non contiene nessun elemento 
     * uguale a <code>obj</code>. 
     * Pi precisamente, se <code>obj</code>  <code>null</code>, 
     * calcola l'indice del primo elemento <code>null</code>, 
     * altrimenti calcola l'indice del primo elemento uguale 
     * (<code>equals</code>) a <code>obj</code>. 
     */ 
    public int indexOf(Object obj) { 
        int index;    // indice di obj nella lista 
        int i;        // per la scansione di data 

        /* cerca obj tra i primi size elementi di data */ 
        index = -1;   // ipotizza che data non contiene obj 
        i = 0; 
        while (index==-1 && i<this.size) { 
            if ((obj==null && data[i]==null) || 
                (obj!=null && obj.equals(data[i]))) 
                index = i;   // trovato obj in posizione i 
            i++; 
        } 

        return index;  
    } 

    /** 
     * Calcola lindice dell'ultimo elemento di questa 
     * lista uguale a <code>obj</code>, oppure <code>-1</code> 
     * se la lista non contiene nessun elemento 
     * uguale a <code>obj</code>. 
     * Pi precisamente, se <code>obj</code>  <code>null</code>, 
     * calcola l'indice dell'ultimo elemento <code>null</code>, 
     * altrimenti calcola l'indice dell'ultimo elemento 
     * uguale a <code>obj</code>. */ 
    public int lastIndexOf(Object obj) { 
        int index;    // indice di obj nella lista 
        int i;        // per la scansione di data 
        
        /* cerca obj tra i primi size elementi di data */ 
        index = -1;   // ipotizza che data non contiene obj 
        i = this.size-1; 
        while (index==-1 && i>=0) { 
            if ((obj==null && data[i]==null) || 
                (obj!=null && obj.equals(data[i]))) 
                index = i;   // trovato obj in posizione i 
            i--; 
        } 
        
        return index; 
    } 
    
    /** Restituisce una descrizione di questa lista. */ 
    public String toString() { 
        String d;    // la descrizione della lista 
        Iterator i;  // per scandire la lista 

        /* apre la parentesi quadra */ 
        d = "["; 

        /* prepara l'iterazione */ 
        i = this.iterator(); 

        /* gestisce il primo elemento (se c') */ 
        if (i.hasNext()) 
            d += i.next().toString(); 

        /* gestisce gli altri elementi, facendoli 
         * precedere da virgola */ 
        while (i.hasNext()) 
            d += ", " + i.next().toString(); 

        /* chiude la parentesi quadra */ 
        d += "]"; 

        return d; 
    } 

    /** 
     * Un Itr  un iteratore per oggetti ArrayList. 
     */ 
    class Itr implements Iterator {     
        /** Indice del prossimo elemento da visitare. */ 
        private int next; 

        /** 
         * Indice dell'ultimo elemento visitato con next(). 
         * Vale -1 se non  stato ancora visitato nessun 
         * elemento, oppure se l'ultimo elemento visitato 
         *  stato cancellato con remove(). 
         */ 
        private int lastSeen; 
    
        /** 
         * Crea un Itr per iterare sulla lista sottostante. 
         */ 
        public Itr() { 
            /* il primo elemento da visitare  il primo 
             * della lista sottostante */ 
            this.next = 0; 
            /* nessun elemento  stato ancora visitato */ 
            this.lastSeen = -1; 
        } 
    
        /** Verifica se ci sono altri elementi da visitare. */ 
        public boolean hasNext() { 
            /* ci sono altri elementi da visitare se this.next 
             *  una posizione valida nella lista sottostante */ 
            return this.next < ArrayList.this.size(); 
        } 
    
        /** Restituisce il prossimo elemento della lista. */ 
        public Object next() { 
            // pre: this.hasNext() 
            Object obj;   // elemento da visitare 

            /* visita l'elemento della lista sottostante 
             * di posizione next */ 
            obj = ArrayList.this.get(next); 
            lastSeen = next; 
            /* prepara la visita del prossimo elemento */ 
            next++; 

            return obj; 
        } 
    
        /** 
         * Rimuove dalla lista l'ultimo elemento 
         * visitato con questo iteratore (opzionale). 
         */ 
        public void remove() { 
            // pre: this.lastSeen!=-1 

            /* rimuove l'ultimo elemento visitato 
             * dalla lista sottostante */ 
            ArrayList.this.remove(lastSeen); 
            /* evita che venga rimosso nuovamente */
            lastSeen = -1; 
            /* la posizione degli elementi a destra di 
             * quello cancellato  stata decrementata di uno, 
             * e quindi anche quella del prossimo elemento 
             * da visitare */ 
            next--; 
        } 
    } 

} 