/*
 * @(#)LinkedList.java    0.1 2000/05/25
 *
 * Luca Cabibbo 
 * Dipartimento di Informatica e Automazione 
 * Universit degli Studi Roma Tre 
 * cabibbo@dia.uniroma3.it 
 */ 

package fiji.util; 
import java.util.Iterator; 

/** 
 * Una <code>LinkedList</code>  una lista implementata 
 * mediante una rappresentazione doppiamente collegata. 
 * ( una implementazione parziale dell'interfaccia 
 * <code>java.util.List</code>.)
 * Ciascun elemento della lista  gestito mediante 
 * un oggetto <code>Entry</code>, che referenzia l'elemento della lista, 
 * nonch riferimenti (agli <code>Entry</code> associati) 
 * agli elementi predecessore e successore. 
 * L'inserimento (cancellazione) di un elemento nella (dalla) lista 
 * comporta l'inserimento (cancellazione) di un oggetto 
 * <code>Entry</code> che lo rappresenta. 
 * 
 * @version 0.1 
 * @author Luca Cabibbo 
 */ 
public class LinkedList implements List { 
    
    /** 
     * Riferimento al primo elemento (testa) della lista. 
     * <code>null</code> se la lista  vuota. 
     */ 
    private Entry head; 
    
    /** 
     * Riferimento all'ultimo elemento (coda) della lista. 
     * <code>null</code> se la lista  vuota. 
     */ 
    private Entry tail; 

    /** 
     * Lunghezza della lista. 
     */ 
    private int size; 
     
    /** 
     * Crea una nuova <code>LinkedList</code>, inizialmente vuota. 
     */ 
    public LinkedList() { 
        this.head = null; 
        this.tail = null; 
        this.size = 0; 
    } 
    
    /* operazioni di base */ 
    
    /** Calcola la lunghezza di questa lista. */ 
    public int size() { 
        return this.size; 
    } 

    /** Verifica se questa lista  vuota. */ 
    public boolean isEmpty() { 
        /* la lista  vuota se ha lunghezza zero */ 
        return this.size==0; 
    } 

    /** 
     * Verifica se questa lista contiene almeno un 
     * elemento uguale a <code>obj</code>. 
     * Pi precisamente, se <code>obj</code>  <code>null</code>, 
     * verifica se questa lista contiene 
     * un elemento <code>null</code>, 
     * altrimenti verifica se contiene un elemento 
     * uguale a <code>obj</code>. 
     */ 
    public boolean contains(Object obj) { 
        return this.indexOf(obj)>=0; 
    } 

    /** 
     * Inserisce <code>obj</code> come ultimo elemento 
     * di questa lista (opzionale). 
     * Restituisce <code>true</code>, perch la lista viene 
     * sicuramente modificata dall'inserimento. 
     */ 
    public boolean add(Object obj) { 
        /* inserisce dopo l'ultimo elemento */ 
        this.addAfter(obj, this.tail); 
        return true; 
    } 
    
    /** 
     * Rimuove da questa lista il primo elemento 
     * uguale a <code>obj</code>, se presente (opzionale). 
     * Restituisce <code>true</code> se un elemento uguale 
     * a <code>obj</code> 
     *  stato effettivamente rimosso dalla lista. 
     */ 
    public boolean remove(Object obj) {
        Entry e;            // per la scansione della lista 
        boolean cancellato; // l'elemento  stato trovato e cancellato
        
        /* cerca il primo elemento della lista uguale a obj 
         * per cancellarlo */ 
        e = this.head; 
        cancellato = false; 
        while (!cancellato && e!=null) { 
            if ((obj==null && e.element==null) || 
                (obj!=null && obj.equals(e.element))) { 
                /* trovato, lo cancella */ 
                this.remove(e); 
                cancellato = true; 
            } else // se non  stato trovato, prosegue la scansione 
                e = e.next; 
        } 
        return cancellato; 
    } 
    
    /** 
     * Restituisce un iteratore sugli elementi di questa 
     * lista, dal primo verso l'ultimo. 
     */ 
    public Iterator iterator() { 
        /* restituisce un iteratore su questa lista */ 
        return new Itr(); 
    } 
    
    /** 
     * Verifica se questa lista contiene tutti gli 
     * elementi della collezione <code>c</code>. 
     */ 
    public boolean containsAll(Collection c) { 
        // pre: c!=null 
        Iterator i;             // per la scansione di c 
        boolean contieneTutti;  // questa lista contiene 
                                // tutti gli elementi di c 
        
        /* verifica se ogni elemento di c  contenuto 
         * nella lista */ 
        contieneTutti = true; 
        i = c.iterator(); 
        while (contieneTutti && i.hasNext()) 
            if (!this.contains(i.next())) 
                contieneTutti = false; 

        return contieneTutti; 
    } 

    /** 
     * Inserisce tutti gli elementi di <code>c</code> in coda 
     * a questa lista, nell'ordine in cui gli elementi
     * vengono visitati dall'iteratore di <code>c</code> (opzionale).
     * Restituisce <code>true</code> se questa lista viene modificata 
     * dall'inserimento. */ 
    public boolean addAll(Collection c) { 
        // pre: c!=null 
        Iterator i;          // per la scansione di c 
        boolean modificata;  // questa lista 
                             //  stata modificata 

        /* itera su c, inserendo gli elementi 
         * in questa lista */ 
        i = c.iterator(); 
        modificata = false;   // finora nessun inserimento 
        while (i.hasNext()) 
            modificata = this.add(i.next()) || modificata; 

        return modificata; 
    }
    
    /** 
     * Rimuove da questa lista tutti gli elementi 
     * della collezione <code>c</code> (opzionale). 
     * Restituisce <code>true</code> se questa lista viene modificata 
     * dalla rimozione. */ 
    public boolean removeAll(Collection c) { 
        // pre: c!=null 
        Iterator i;          // per la scansione di c 
        boolean modificata;  // la lista  stata modificata 
        
        /* itera su c rimuovendo gli elementi dalla lista */ 
        i = c.iterator(); 
        modificata = false;   // finora nessuna rimozione 
        while (i.hasNext()) 
            modificata = this.remove(i.next()) ||
                         modificata; 
                         
        return modificata; 
    }
    
    /** 
     * Rimuove da questa lista tutti gli elementi che non 
     * sono contenuti nella collezione <code>c</code> (opzionale).
     * Restituisce <code>true</code> se questa lista viene modificata 
     * dalla rimozione. 
     */ 
    public boolean retainAll(Collection c) { 
        // pre: c!=null 
        Iterator i;          // per la scansione della lista
        boolean modificata;  // questa lista 
                             //  stata modificata 

        /* itera su questa lista, rimuovendo gli elementi 
         * che non sono contenuti in c */ 
        i = this.iterator(); 
        modificata = false;   // finora nessuna rimozione 
        while (i.hasNext()) 
            /* se c non contiene l'elemento corrente */ 
            if (!c.contains(i.next())) { 
                /* rimuovilo, usando l'iteratore */ 
                i.remove(); 
                modificata = true; 
            } 

        return modificata; 
    }
    
    /** 
     * Rimuove tutti gli elementi da questa lista 
     * (opzionale). 
     */ 
    public void clear() { 
        this.head = null; 
        this.tail = null; 
        this.size = 0; 
    } 
    
    /** 
     * Restituisce un array che contiene gli elementi di 
     * questa lista, ciascuno nella posizione in cui compare 
     * nella lista. 
     */ 
    public Object[] toArray() { 
        Object[] result;   // array da creare e restituire 
        Iterator i;        // per la scansione della lista 
        int index;         // per la scansione della lista e di result 

        /* scandisce i size elementi della lista 
         * e li memorizza in result */ 
        result = new Object[this.size]; 
        i = this.iterator(); 
        for (index=0; index<this.size; index++) 
            result[index] = i.next(); 
            
        return result; 
    } 
    
    /** 
     * verifica se questa lista  uguale a <code>obj</code>. 
     * Pi precisamente, restituisce <code>true</code> 
     * se anche <code>obj</code> 
     *  una lista e le due liste sono uguali, 
     * ovvero hanno la stessa lunghezza e ogni elemento 
     * di questa lista  uguale a quello che occupa la 
     * stessa posizione in <code>obj</code>. 
     */ 
    public boolean equals(Object obj) { 
        boolean uguali;   // questa lista e obj sono uguali 
        Iterator i1, i2;  // per la scansione di questa
                          // lista e di obj (se lista) 
        Object o1, o2;    // elementi di questa lista 
                          // e di obj (se lista) 

        if (this==obj)     // se sono lo stesso oggetto 
            uguali = true;     // allora sono uguali 
        else if (obj instanceof List) { 
                           // se anche obj  una lista 
            /* anche obj  una lista */ 
            i1 = iterator();   // per scandire questa lista 
            i2 = ((List) obj).iterator(); //per scandire obj
            uguali = true; //sono uguali se non sono diverse 
            while (i1.hasNext() && i2.hasNext()) { 
                o1 = i1.next(); 
                o2 = i2.next(); 
                // se o1 non  uguale a o2 
                if (!(o1==null && o2==null) && 
                    !(o1!=null && o1.equals(o2))) 
                   uguali = false; 
            } 
            /* questa lista e obj sono uguali se sono state 
             * entrambe scandite completamente senza trovare 
             * differenze */ 
            uguali = uguali && 
                     !i1.hasNext() && !i2.hasNext(); 
        } else   // obj non  una lista 
            uguali = false; 

        return uguali; 
    } 
    
    /** 
     * Restituisce l'elemento di questa lista 
     * di posizione <code>index</code>. 
     */ 
    public Object get(int index) { 
        // pre: index>=0 && index<this.size 
        /* l'elemento cercato  quello associato alla 
         * Entry di indice index */ 
        return entry(index).element; 
    } 
    
    /** 
     * Sostituisce l'elemento di questa lista 
     * di posizione <code>index</code> con <code>obj</code>. 
     * Restituisce l'elemento che precedentemente 
     * occupava la posizione <code>index</code>. 
     */ 
    public Object set(int index, Object obj) { 
        // pre: index>=0 && index<this.size 
        Entry e;        // la Entry associata all'elemento di indice index 
        Object oldObj;  // il valore precedente dell'elemento 
        
        /* l'elemento cercato  quello associato alla 
         * Entry di indice index */ 
        e = entry(index); 
        /* memorizza il valore precedente */ 
        oldObj = e.element; 
        /* aggiorna l'elemento */ 
        e.element = obj; 
        /* restituisce il valore precedente */ 
        return oldObj; 
    } 

    /** 
     * Inserisce in questa lista l'elemento <code>obj</code> 
     * nella posizione <code>index</code>. 
     * Tutti gli elementi che precedentemente occupavano 
     * una posizione maggiore o uguale a <code>index</code> 
     * vengono spostati di una posizione verso destra. 
     */ 
    public void add(int index, Object obj) { 
        // pre: index>=0 && index<=this.size 
        Entry e;        // la Entry associata all'elemento di indice index
        
        /* inserisce obj in posizione index */ 
        if (index==this.size)   // se  un inserimento in coda 
            this.add(obj); 
        else {       // inserisce prima dell'elemento di indice index 
            e = entry(index); 
            this.addBefore(obj, e); 
        } 
    } 

    /** 
     * Rimuove da questa lista l'elemento di posizione 
     * <code>index</code> e lo restituisce. 
     * Tutti gli elementi che precedentemente occupavano 
     * una posizione maggiore di <code>index</code> vengono 
     * spostati di una posizione verso sinistra. 
     */ 
    public Object remove(int index) { 
        // pre: index>=0 && index<this.size 
        Entry e;     // la Entry di posizione index 
        Object obj;  // elemento di posizione index 
        
        /* accede all'elemento di posizione index */ 
        e = entry(index); 
        obj = e.element; 
        
        /* lo rimuove, e restituisce il suo valore */ 
        this.remove(e); 
        return obj; 
    } 
    
    /** 
     * Inserisce in questa lista tutti gli elementi 
     * contenuti nella collezione <code>c</code>, 
     * a partire dalla posizione <code>index</code> (opzionale). 
     * L'inserimento avviene nell'ordine in cui gli elementi 
     * di <code>c</code> vengono visitati dal suo iteratore. 
     * Restituisce <code>true</code> questa lista 
     *  stata modificata dall'inserimento. 
     */ 
    public boolean addAll(int index, Collection c) { 
        // pre: index>=0 && index<=this.size() 
        Iterator i;           // per la scansione di c; 
        boolean modificata;   // questa lista  stata modificata 
        
        /* inserisce tutti gli elementi di c in coda alla lista, 
         * come restituiti dall'iteratore di c */ 
        modificata = false;  
        i = c.iterator(); 
        while (i.hasNext()) { 
            this.add(index, i.next()); 
            modificata = true; 
            index++;  // il prossimo sar inserito alla destra 
                      // di quest'ultimo inserito 
        } 
                
        return modificata; 
    } 
    
    
    /** 
     * Calcola lindice del primo elemento di questa lista
     * uguale a <code>obj</code>, oppure <code>-1</code> 
     * se la lista non contiene nessun elemento 
     * uguale a <code>obj</code>. 
     * Pi precisamente, se <code>obj</code>  <code>null</code>, 
     * calcola l'indice del primo elemento <code>null</code>, 
     * altrimenti calcola l'indice del primo elemento uguale 
     * (<code>equals</code>) a <code>obj</code>. 
     */ 
    public int indexOf(Object obj) { 
        int index;    // indice di obj nella lista 
        Entry e;      // per scandire la lista 
        int i;        // indice dell'elemento corrente  

        index = -1;   // finora non  stato incontrato 
        /* si posizione sul primo */ 
        e = this.head; 
        i = 0; 
        /* scandisce gli elementi della lista */ 
        while (index==-1 && e!=null) { 
            if ((obj==null && e.element==null) || 
                (obj!=null && obj.equals(e.element))) 
                index = i;   // trovato obj in posizione i 
            else { //prosegue la scansione 
                e = e.next; 
                i++; 
            } 
        } 

        return index;  
    } 

    /** 
     * Calcola lindice dell'ultimo elemento di questa 
     * lista uguale a <code>obj</code>, oppure <code>-1</code> 
     * se la lista non contiene nessun elemento 
     * uguale a <code>obj</code>. 
     * Pi precisamente, se <code>obj</code>  <code>null</code>, 
     * calcola l'indice dell'ultimo elemento <code>null</code>, 
     * altrimenti calcola l'indice dell'ultimo elemento 
     * uguale a <code>obj</code>. */ 
    public int lastIndexOf(Object obj) { 
        int index;    // indice di obj nella lista 
        Entry e;      // per scandire la lista 
        int i;        // indice dell'elemento corrente  

        index = -1;   // finora non  stato incontrato 
        /* si posizione sull'ultimo */ 
        e = this.tail; 
        i = this.size-1; 
        /* scandisce gli elementi della lista */ 
        while (index==-1 && e!=null) { 
            if ((obj==null && e.element==null) || 
                (obj!=null && obj.equals(e.element))) 
                index = i;   // trovato obj in posizione i 
            else { //prosegue la scansione 
                e = e.previous; 
                i--; 
            } 
        } 

        return index;  
    } 
    
    /** Restituisce una descrizione di questa lista. */ 
    public String toString() { 
        String d;    // la descrizione della lista 
        Iterator i;  // per scandire la lista 

        /* apre la parentesi quadra */ 
        d = "["; 

        /* prepara l'iterazione */ 
        i = this.iterator(); 

        /* gestisce il primo elemento (se c') */ 
        if (i.hasNext()) 
            d += i.next().toString(); 

        /* gestisce gli altri elementi, facendoli 
         * precedere da virgola */ 
        while (i.hasNext()) 
            d += ", " + i.next().toString(); 

        /* chiude la parentesi quadra */ 
        d += "]"; 

        return d; 
    } 
    
    /* La classe Entry e operazioni di gestione di Entry. */ 
    
    /** 
     * Un oggetto Entry rappresenta un elemento della lista, 
     * nonch riferimenti agli elementi predecessore e successore. 
     */ 
    private static class Entry {         
        /** Elemento della lista. */ 
        public Object element; 

        /** Riferimento al prossimo elemento della lista. */ 
        public Entry next; 

        /** Riferimento al precedente elemento della lista. */ 
        public Entry previous; 
        
        public Entry(Object element, Entry next, Entry previous) { 
            this.element = element; 
            this.next = next; 
            this.previous = previous; 
        } 
    } 
    
    /** Restituisce la Entry di indice index. */ 
    private Entry entry(int index) { 
        // pre: index>=0 && index<this.size 
        Entry e;    // la Entry cercata 
        int i;      // per contare gli elementi scanditi 

        /* scandisce la lista dalla testa verso la coda 
         * o viceversa minimizzando il numero di 
         * elementi scanditi */ 
        if (index<size/2) {  // pi vicino alla testa 
            /* scandisce dalla testa verso la coda */ 
            e = this.head; 
            /* deve scandire index elementi */ 
            for (i=0; i<index; i++) 
                e = e.next; 
        } else {             // pi vicino alla coda 
            /* scandisce dalla coda verso la testa */ 
            e = this.tail; 
            /* deve scandire size-1-index elementi */ 
            for (i=1; i<size-index; i++) 
                e = e.previous; 
        } 
        
        return e; 
    } 
    
    /** 
     * Inserisce in questa lista l'elemento obj 
     * dopo la Entry e, e restituisce la Entry creata. 
     * Se la lista  vuota, e vale null, e 
     * l'elemento viene inserito come unico elemento 
     * della lista. 
     */ 
    private Entry addAfter(Object obj, Entry e) { 
        Entry newEntry;   // la Entry creata 
        
        if (e==null) {  // inserimento nella lista vuota 
            newEntry = new Entry(obj, null, null); 
            this.head = newEntry; 
            this.tail = newEntry; 
        } else {       // inserimento in una lista non vuota 
            newEntry = new Entry(obj, e.next, e); 
            e.next = newEntry; 
            if (newEntry.next==null)  // se  l'ultimo
                this.tail = newEntry; 
            else                      // se non  l'ultimo 
                newEntry.next.previous = newEntry; 
        } 
        /* aggiorna la lunghezza della lista */ 
        this.size++; 
        return newEntry; 
    } 

    /** 
     * Inserisce in questa lista l'elemento obj 
     * prima della Entry e, e restituisce la Entry creata. 
     *
     * Se la lista  vuota, e vale null, e l'elemento 
     * inserito come unico elemento della lista. 
     * Di norma, viene invocato con e!=null; 
     * tuttavia, l'implementazione considera anche 
     * il caso e==null per simmetria rispetto a addAfter. 
     */ 
    private Entry addBefore(Object obj, Entry e) { 
        Entry newEntry;   // la Entry creata 
        
        if (e==null) {  // inserimento nella lista vuota 
            newEntry = new Entry(obj, null, null); 
            this.head = newEntry; 
            this.tail = newEntry; 
        } else {        // inserimento in una lista non vuota 
            newEntry = new Entry(obj, e, e.previous); 
            e.previous = newEntry; 
            if (newEntry.previous==null)  // se  il primo 
                this.head = newEntry; 
            else                          // non  il primo 
                newEntry.previous.next = newEntry; 
        } 

        /* aggiorna la lunghezza della lista */ 
        this.size++; 
        return newEntry; 
    } 
    
    /** 
     * Inserisce in questa lista l'elemento obj 
     * prima della Entry e (non nulla), 
     * e restituisce la Entry creata. 
     */ 
    private Entry addBeforeSemplificato(Object obj, Entry e) { 
        // pre: e!=null 
        Entry newEntry;   // la Entry creata 
        
        /* l'inserimento avviene in una lista non vuota, 
         * prima di una entry non null */ 
        newEntry = new Entry(obj, e, e.previous); 
        e.previous = newEntry; 
        if (newEntry.previous==null)  // se  il primo 
            this.head = newEntry; 
        else                          // se non  il primo 
            newEntry.previous.next = newEntry; 
        /* aggiorna la lunghezza della lista */ 
        this.size++; 
        return newEntry; 
    } 

    /** 
     * Rimuove da questa lista la Entry e. 
     */ 
    private void remove(Entry e) { 
        // pre: e!=null 
        
        /* aggiorna il successore di e */ 
        if (e.next!=null) // se non  l'ultima entry 
            e.next.previous = e.previous; 
        else              //  la coda 
            this.tail = e.previous; 
        /* aggiorna il predecessore di e */ 
        if (e.previous!=null) // se non  la prima entry 
            e.previous.next = e.next; 
        else                  //  la testa 
            this.head = e.next; 

        /* aggiorna la lunghezza della lista */ 
        this.size--; 
    } 

    /* La classe Itr. */ 
    
    /** 
     * Un Itr  un iteratore per oggetti LinkedList. 
     */ 
    private class Itr implements Iterator { 
        /** Il prossimo elemento da visitare. */ 
        private Entry next; 

        /** 
         * Ultimo elemento visitato con next(). 
         * Vale null se non  stato ancora visitato nessun 
         * elemento, oppure se l'ultimo elemento visitato 
         *  stato cancellato con remove(). 
         */ 
        private Entry lastSeen; 
    
        /** 
         * Crea un Itr per iterare sulla lista sottostante. 
         */ 
        public Itr() { 
            /* il primo elemento da visitare  il primo 
             * della lista sottostante*/ 
            this.next = LinkedList.this.head; 
            /* nessun elemento  stato ancora visitato */ 
            this.lastSeen = null; 
        } 
    
        /** Verifica se ci sono altri elementi da visitare. */ 
        public boolean hasNext() { 
            /* ci sono altri elementi da visitare 
             * se this.next referenzia una Entry */ 
            return this.next!=null; 
        } 
    
        /** Restituisce il prossimo elemento della lista. */ 
        public Object next() { 
            // pre: this.hasNext() 
            Object obj;   // elemento da visitare 

            /* visita la Entry next della lista sottostante */ 
            obj = next.element; 
            lastSeen = next; 
            /* prepara la visita del prossimo elemento */ 
            next = next.next; 

            return obj; 
        } 
    
        /** 
         * Rimuove dalla lista l'ultimo elemento 
         * visitato con questo iteratore (opzionale). 
         */ 
        public void remove() { 
            // pre: this.lastSeen!=null 

            /* rimuove la Entry lastSeen dalla lista sottostante */ 
            LinkedList.this.remove(lastSeen);  
            /* evita che venga rimosso nuovamente */
            lastSeen = null; 
        }     
    } 
    
} 