package it.unibo.tw.es2.persistence;

import it.unibo.tw.es2.beans.DescrizioneMacchina;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;

public class Db2MacchinaDAO {

	/**
	 * Name of the class that holds the jdbc driver implementation for the DB2 database
	 */
	public static final String DRIVER = "com.ibm.db2.jcc.DB2Driver";
	
	/**
	 * URI of the database to connect to
	 */
	public static final String DBURL = "jdbc:db2://diva.deis.unibo.it:50000/tw_stud";

	public static final String USERNAME = "username";
	public static final String PASSWORD = "password";

	static final String TABLE = "macchine";
	
	// -------------------------------------------------------------------------------------
	
	static final String ID = "id";
	static final String COLOUR = "colore";
	static final String MODEL = "modello";
	static final String PLATE = "targa";
	
	// == STATEMENT SQL ====================================================================
	
	// INSERT INTO table ( id, name, description, ...) VALUES ( ?,?, ... );
	static final String insert = 
			"INSERT " +
					"INTO " + TABLE + " ( " + 
					ID+", "+COLOUR+", "+MODEL+", "+PLATE+" " +
					") " +
					"VALUES ((SELECT COALESCE(MAX("+ID+")+1,1) FROM "+TABLE+"),?,?,?) "
					;
	
	// SELECT * FROM table WHERE idcolumn = ?;
	static final String read_by_plate = 
			"SELECT * " +
					"FROM " + TABLE + " " +
					"WHERE " + PLATE + " = ? "
					;
	
	// -------------------------------------------------------------------------------------
	
	// CREATE entrytable ( code INT NOT NULL PRIMARY KEY, ... );
	static final String create = 
			"CREATE " +
					"TABLE " + TABLE +" ( " +
					ID + " INT NOT NULL PRIMARY KEY, " +
					COLOUR + " VARCHAR(50), " +
					MODEL + " VARCHAR(50), " +
					PLATE + " VARCHAR(10) " +
					") "
					;
	
	// --------------------------------------------

	// static initializer block to load db driver class in memory
	static {
		try {
			Class.forName(DRIVER);
		} 
		catch (Exception e) {
			System.err.println("HsqldbDAOFactory: failed to load DB2 JDBC driver" + "\n" + e.toString());
			e.printStackTrace();
		}
	}

	// --------------------------------------------

	/**
	 * For the sake of simplicity, here we create a brand new connection every time we are asked to.
	 * 
	 * Anyway, this is how things should go in the real world:
	 * 1. 	DB offers a limited number of connections (typically 100) so we can't really afford to 
	 * 		rely on factory invokers to always terminate the connection we have opened
	 * 2. 	At initialization, factory creates a well-defined number of connections (say 20) and put them
	 * 		in a pool
	 * 3.	create and close connection methods actually do not create or close brand new connections, but
	 * 		pick available connections from the pool, each time, and mark them as 'busy'/'free'
	 * 4.	factory does monitoring on the connection pool and force termination of connection in use
	 * 		for a too long time (invoker prob'ly forgot to close it... maybe due to exceptions it didn't handle well)
	 * 5.	terminated connections are replaced by brand new ones
	 */
	
	private Connection createConnection() {
		try {
			return DriverManager.getConnection(DBURL,USERNAME,PASSWORD);
		} 
		catch (Exception e) {
			System.err.println(Db2MacchinaDAO.class.getName() + ".createConnection(): failed creating connection" + "\n" + e.toString());
			e.printStackTrace();
			System.err.println("Was the database started? Is the database URL right?");
			return null;
		}
	}
	
	/**
	 * For the sake of simplicity, here we actually close the connection we are told to.
	 */
	private void closeConnection(Connection conn) {
		try {
			conn.close();
		}
		catch (Exception e) {
			System.err.println(Db2MacchinaDAO.class.getName() + ".closeConnection(): failed closing connection" + "\n" + e.toString());
			e.printStackTrace();
		}
	}
	
	// -------------------------------------------------------------------------------------
	
	public void create(String t, DescrizioneMacchina d) {
		
		Connection conn = createConnection();
		try {
			PreparedStatement prep_stmt = conn.prepareStatement(insert);
			prep_stmt.clearParameters();
			prep_stmt.setString(1, d.getColore());
			prep_stmt.setString(2, d.getModello());
			prep_stmt.setString(3, t);
			prep_stmt.executeUpdate();
			prep_stmt.close();
		}
		catch (Exception e) {
			e.printStackTrace();
		}
		finally {
			closeConnection(conn);
		}
	}

	public DescrizioneMacchina getMacchina(String targa) {
		
		DescrizioneMacchina result = null;
		
		Connection conn = createConnection();

		try {
			PreparedStatement prep_stmt = conn.prepareStatement(read_by_plate);
			prep_stmt.clearParameters();
			prep_stmt.setString(1, targa);
			ResultSet rs = prep_stmt.executeQuery();
			if ( rs.next() ) {
				result = new DescrizioneMacchina(rs.getString(MODEL),rs.getString(COLOUR));
			}
			rs.close();
			prep_stmt.close();
		}
		catch (Exception e) {
			e.printStackTrace();
		}
		finally {
			closeConnection(conn);
		}
		return result;
	}

	public boolean createTable() {
		
		boolean result = false;

		Connection conn = createConnection();

		try {

			Statement stmt = conn.createStatement();
			stmt.execute(create);
			result = true;
			stmt.close();
		}
		catch (Exception e) {
			e.printStackTrace();
		}
		finally {
			closeConnection(conn);
		}
		return result;
	}
	
}
